// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_POSIX_RW_LOCK_HH__
#define __TPL_POSIX_RW_LOCK_HH__

#include "libtpl/tpl_decls.hh"
#include <pthread.h>

#if defined __USE_UNIX98 || defined __USE_XOPEN2K

namespace tpl {

/**
 *  @short POSIX read/write lock attributes wrapper class.
 */
class PosixRWLockAttributes
{
public:
    PosixRWLockAttributes(int kind, int shared = PTHREAD_PROCESS_PRIVATE);

    ~PosixRWLockAttributes();

    TPL_NO_COPY(PosixRWLockAttributes);

public:  // State

    /**
     * @return PTHREAD_RWLOCK_PREFER_READER_NP, PTHREAD_RWLOCK_PREFER_WRITER_NP
     * or PTHREAD_RWLOCK_PREFER_WRITER_NONRECURSIVE_NP
     * values of the kind attribute.
     */
    int kind() const;

    /**
     * @return PTHREAD_PROCESS_PRIVATE or PTHREAD_PROCESS_SHARED
     * values of the sharing attribute.
     */
    int shared() const;

    /**
     *	@return reference to the internal representation of rwlockattr.
     */
    const pthread_rwlockattr_t& rwlockattr_handle() const;

public:  // Constants

    static const PosixRWLockAttributes PREFER_READER;
    static const PosixRWLockAttributes PREFER_WRITER;
    static const PosixRWLockAttributes PREFER_WRITER_NONRECURSIVE;

private:
    pthread_rwlockattr_t _attr;
};

/**
 *  @short POSIX read/write lock facility wrapper class
 */
template <const PosixRWLockAttributes& ATTRIBUTES>
class PosixRWLock
{
public:  // Create/Copy/Destroy

    PosixRWLock();

    virtual ~PosixRWLock();

    TPL_NO_COPY_TEMPLATE(PosixRWLock, ATTRIBUTES);

public:  // Usage

    /**
     *  Try to acquire read lock
     */
    bool try_read_lock() const;

    /**
     *  Acquire read lock
     */
    void read_lock() const;

    /**
     *  Try to acquire write lock
     */
    bool try_write_lock() const;

    /**
     *  Acquire write lock
     */
    void write_lock() const;

    /**
     *  Release lock
     */
    void unlock() const;

public:  // State

    /**
     * @return PTHREAD_RWLOCK_PREFER_READER_NP, PTHREAD_RWLOCK_PREFER_WRITER_NP
     * or PTHREAD_RWLOCK_PREFER_WRITER_NONRECURSIVE_NP
     * values of the kind attribute
     */
    int kind() const;

    /**
     *  @return true if the read/write lock is shared
     */
    bool is_shared() const;

    /**
     *  @return reference to the internal representation of read/write lock
     */
    const pthread_rwlock_t& rwlock_handle() const;

private:

    mutable pthread_rwlock_t _rwlock;
};

};  // namespace tpl

#endif //__USE_UNIX98 || __USE_XOPEN2K

#endif //__TPL_POSIX_RW_LOCK_HH__
