/*
 *
 * Copyright (C) 2002
 * ALL RIGHTS RESERVED
 *
 */

#include <errno.h>
#include "posix_rw_lock.hh"
#include "libtpl/debug.hh"

#include <iostream>

#if defined __USE_UNIX98 || defined __USE_XOPEN2K

using namespace tpl;

////////////////////////////////////////////////////////////
//              PosixRWLockAttributes                     //
////////////////////////////////////////////////////////////

const PosixRWLockAttributes PosixRWLockAttributes::PREFER_READER(PTHREAD_RWLOCK_PREFER_READER_NP);
const PosixRWLockAttributes PosixRWLockAttributes::PREFER_WRITER(PTHREAD_RWLOCK_PREFER_WRITER_NP);
const PosixRWLockAttributes
  PosixRWLockAttributes::PREFER_WRITER_NONRECURSIVE(PTHREAD_RWLOCK_PREFER_WRITER_NONRECURSIVE_NP);

PosixRWLockAttributes::PosixRWLockAttributes(int kind, int shared)
{
    TPL_VERIFY(::pthread_rwlockattr_init(&_attr) == 0);

    TPL_VERIFY(::pthread_rwlockattr_setpshared(&_attr, shared) == 0);
    TPL_VERIFY(::pthread_rwlockattr_setkind_np(&_attr, kind) == 0);
}

PosixRWLockAttributes::~PosixRWLockAttributes()
{
    TPL_VERIFY(::pthread_rwlockattr_destroy(&_attr) == 0);
}

/**
 * @return PTHREAD_RWLOCK_PREFER_READER_NP, PTHREAD_RWLOCK_PREFER_WRITER_NP
 * or PTHREAD_RWLOCK_PREFER_WRITER_NONRECURSIVE_NP
 * values of the kind attribute.
 */
int
PosixRWLockAttributes::kind() const
{
    int k = PTHREAD_RWLOCK_DEFAULT_NP;
    TPL_VERIFY(::pthread_rwlockattr_getkind_np(&_attr, &k) == 0);
    return k;
}

/**
 * @return PTHREAD_PROCESS_PRIVATE or PTHREAD_PROCESS_SHARED
 * values of the sharing attribute.
 */
int
PosixRWLockAttributes::shared() const
{
    int s = PTHREAD_PROCESS_PRIVATE;
    TPL_VERIFY(::pthread_rwlockattr_getpshared(&_attr, &s) == 0);
    return s;
}

const pthread_rwlockattr_t&
PosixRWLockAttributes::rwlockattr_handle() const
{
    return _attr;
}

////////////////////////////////////////////////////////////
//                      PosixMutex                        //
////////////////////////////////////////////////////////////

template <const PosixRWLockAttributes& ATTRIBUTES>
PosixRWLock<ATTRIBUTES>::PosixRWLock()
{
    TPL_VERIFY(::pthread_rwlock_init(&_rwlock, &(ATTRIBUTES.rwlockattr_handle())) == 0);
}

template <const PosixRWLockAttributes& ATTRIBUTES>
PosixRWLock<ATTRIBUTES>::~PosixRWLock()
{
    TPL_VERIFY(::pthread_rwlock_destroy(&_rwlock) == 0);
}

template <const PosixRWLockAttributes& ATTRIBUTES>
bool
PosixRWLock<ATTRIBUTES>::try_read_lock() const
{
    int err = ::pthread_rwlock_tryrdlock(&_rwlock);

    if (err == 0) {
    	return true;
    }

    TPL_ASSERT(err == EBUSY);
    return false;
}

template <const PosixRWLockAttributes& ATTRIBUTES>
void
PosixRWLock<ATTRIBUTES>::read_lock() const
{
    int err;
    if ((err = ::pthread_rwlock_rdlock(&_rwlock)) != 0) {
	TPL_ASSERT(err == EDEADLK);

// TODO: change direct output to cerr and call to abort to some
// centralized termination function
        if (err == EDEADLK) {
            std::cerr << "PosixRWLock: Deadlock detected" << std::endl;
            abort();
        }
    }
}

template <const PosixRWLockAttributes& ATTRIBUTES>
bool
PosixRWLock<ATTRIBUTES>::try_write_lock() const
{
    int err = ::pthread_rwlock_trywrlock(&_rwlock);

    if (err == 0) {
    	return true;
    }

    TPL_ASSERT(err == EBUSY);
    return false;
}

template <const PosixRWLockAttributes& ATTRIBUTES>
void
PosixRWLock<ATTRIBUTES>::write_lock() const
{
    int err;
    if ((err = ::pthread_rwlock_wrlock(&_rwlock)) != 0) {
	TPL_ASSERT(err == EDEADLK);

// TODO: change direct output to cerr and call to abort to some
// centralized termination function
        if (err == EDEADLK) {
            std::cerr << "PosixRWLock: Deadlock detected" << std::endl;
            abort();
        }
    }
}

template <const PosixRWLockAttributes& ATTRIBUTES>
void
PosixRWLock<ATTRIBUTES>::unlock() const
{
    TPL_VERIFY(::pthread_rwlock_unlock(&_rwlock) == 0);
}

template <const PosixRWLockAttributes& ATTRIBUTES>
int
PosixRWLock<ATTRIBUTES>::kind() const
{
    return ATTRIBUTES.kind();
}

template <const PosixRWLockAttributes& ATTRIBUTES>
bool
PosixRWLock<ATTRIBUTES>::is_shared() const
{
    return ATTRIBUTES.shared() == PTHREAD_PROCESS_SHARED;
}

template <const PosixRWLockAttributes& ATTRIBUTES>
const pthread_rwlock_t&
PosixRWLock<ATTRIBUTES>::rwlock_handle() const
{
    return _rwlock;
}

//------------------------------------------------------------
// Template Instantiations

template class PosixRWLock<PosixRWLockAttributes::PREFER_READER>;
template class PosixRWLock<PosixRWLockAttributes::PREFER_WRITER>;
template class PosixRWLock<PosixRWLockAttributes::PREFER_WRITER_NONRECURSIVE>;

#endif //__USE_UNIX98 || __USE_XOPEN2K

