// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_POSIX_SYNC_OBJECTS_HH__
#define __TPL_POSIX_SYNC_OBJECTS_HH__

#include "libtpl/tpl_decls.hh"

#include <pthread.h>

namespace tpl {

#ifdef __USE_XOPEN2K
typedef pthread_spinlock_t tpl_spinlock_t;
#else
typedef struct {
    volatile uint32_t lock;
} tpl_spinlock_t;
#endif

/**
 * @short POSIX spinlock facility - a lightweight locking mechanism for short locks.
 */
template <int TYPE = PTHREAD_PROCESS_PRIVATE>
class PosixSpinlock
{
public:  // Create/Copy/Destroy

    PosixSpinlock();

    ~PosixSpinlock();

    TPL_NO_COPY_TEMPLATE(PosixSpinlock, TYPE);

public:	 // Usage

    /**
     *  Try to acquire spinlock
     *  @return false - if spinlock is currently acquired by another thread 
     *          true - if acquired successfully
     *  @throw ErrorFatal - failed to trylock().
     */
    bool try_lock() const;

    /**
     *  Acquire spinlock
     *  If spinlock is currently acquired by another thread
     *  the calling thread blocks until spinlock becomes available.
     *  @throw ErrorFatal - failed to lock().
     */
    void lock() const;

    /**
     *  Release spinlock.
     *  If any other thread is blocked on this spinlock it becomes
     *  the owner.
     *  @throw ErrorFatal - failed to unlock().
     */
    void unlock() const;

public:  // State

    /**
     * @return true if the mutex is shared.
     */
    bool is_shared() const;

    /**
     *	@return reference to internal spinlock structure.
     */
    const tpl_spinlock_t& spinlock_handle() const;

private:
    mutable tpl_spinlock_t _spinlock;	// Spinlock handle.
};

/**
 *  @short POSIX mutex attributes wrapper class.
 */
class PosixMutexAttributes
{
public:
    PosixMutexAttributes(int kind, int shared = PTHREAD_PROCESS_PRIVATE);

    ~PosixMutexAttributes();

    TPL_NO_COPY(PosixMutexAttributes);

public:  // State

    /**
     * @return PTHREAD_MUTEX_RECURSIVE, PTHREAD_MUTEX_ERRORCHECK or PTHREAD_MUTEX_DEFAULT
     * values of the kind attribute.
     */
    int kind() const;

    /**
     * @return PTHREAD_PROCESS_PRIVATE or PTHREAD_PROCESS_SHARED
     * values of the sharing attribute.
     */
    int shared() const;

    /**
     *	@return reference to the internal representation of mutexattr.
     */
    const pthread_mutexattr_t& mutexattr_handle() const;

public:  // Constants

    static const PosixMutexAttributes NORMAL;
    static const PosixMutexAttributes RECURSIVE;
    static const PosixMutexAttributes ERRORCHECKING;

private:
    pthread_mutexattr_t _attr;
};

/**
 *
 *  @short POSIX mutex facility wrapper class
 *
 */
template <const PosixMutexAttributes& ATTRIBUTES>
class PosixMutex
{
public:  // Create/Copy/Destroy

    PosixMutex();

    ~PosixMutex();

    TPL_NO_COPY_TEMPLATE(PosixMutex, ATTRIBUTES);

public:  // Usage

    /**
     *  Try to acquire mutex
     *  If mutex is currently locked returns immidiatly 
     */
    bool try_lock() const;

    /**
     *  Acquire mutex
     *  If mutex is currently locked the calling thread blocks until 
     *  mutex becomes available. 
     */
    void lock() const;

    /**
     *  Release mutex.
     */
    void unlock() const;

public:  // State

    /**
     *  @return PTHREAD_MUTEX_RECURSIVE, PTHREAD_MUTEX_ERRORCHECK or PTHREAD_MUTEX_DEFAULT
     *  values of the kind attribute.
     */
    int kind() const;

    /**
     *  @return true if the mutex is shared.
     */
    bool is_shared() const;

    /**
     *  @return reference to the internal representation of mutex.
     */
    const pthread_mutex_t& mutex_handle() const;

#ifdef TPL_DEBUG
    /**
     *  @returns current host thread ID or 0 if no threads are posessing the mutex.
     */
    pthread_t owner_thread() const;
#endif // TPL_DEBUG

private:
    mutable pthread_mutex_t _mutex;     // Mutex handle.
#ifdef TPL_DEBUG
    mutable pthread_t _owner_thread;     // Owner thread.
#endif // TPL_DEBUG
};

}; // namespace tpl

#endif //__TPL_POSIX_SYNC_OBJECTS_HH__
