/*
 *
 * Copyright (C) 2002
 * ALL RIGHTS RESERVED
 *
 */

#include <errno.h>
#include "posix_lock_objects.hh"
#include "libtpl/debug.hh"

#include <iostream>

#define __stringify_1(x)    #x
#define __stringify(x)      __stringify_1(x)

using namespace tpl;

////////////////////////////////////////////////////////////
//                   PosixSpinlock                        //
////////////////////////////////////////////////////////////

template <int TYPE>
PosixSpinlock<TYPE>::PosixSpinlock()
{
#ifdef __USE_XOPEN2K
    TPL_VERIFY(::pthread_spin_init(&_spinlock, TYPE) == 0);
#else
    _spinlock = (tpl_spinlock_t) { 1 };
#endif
}

template <int TYPE>
PosixSpinlock<TYPE>::~PosixSpinlock()
{
#ifdef __USE_XOPEN2K
    TPL_VERIFY(::pthread_spin_destroy(&_spinlock) == 0);
#endif
}

template <int TYPE>
bool
PosixSpinlock<TYPE>::try_lock() const
{
#ifdef __USE_XOPEN2K
    int err = ::pthread_spin_trylock(&_spinlock);

    if (err == 0)
    	return true;

    TPL_ASSERT(err == EBUSY);
    return false;
#else
    char oldval;
    __asm__ __volatile__(
          "xchgb %b0,%1"
          :"=q" (oldval), "=m" (_spinlock.lock)
          :"0" (0) : "memory");
    return oldval > 0;
#endif
}

template <int TYPE>
void
PosixSpinlock<TYPE>::lock() const
{
#ifdef __USE_XOPEN2K
    TPL_VERIFY(::pthread_spin_lock(&_spinlock) == 0);
#else
    __asm__ __volatile__(
        "\n1:\t" \
        "lock ; decb %0\n\t" \
        "js 2f\n" \
        ".subsection 1\n" \
        ".ifndef _text_lock_" __stringify(KBUILD_BASENAME) "\n" \
        "_text_lock_" __stringify(KBUILD_BASENAME) ":\n" \
        ".endif\n" \
        "2:\t" \
        "cmpb $0,%0\n\t" \
        "rep;nop\n\t" \
        "jle 2b\n\t" \
        "jmp 1b\n" \
        ".subsection 0\n"
        :"=m" (_spinlock.lock) : : "memory");
#endif
}

template <int TYPE>
void
PosixSpinlock<TYPE>::unlock() const
{
#ifdef __USE_XOPEN2K
    TPL_VERIFY(::pthread_spin_unlock(&_spinlock) == 0);
#else
    __asm__ __volatile__(
        "movb $1,%0" \
        :"=m" (_spinlock.lock) : : "memory"
    );
#endif
}

template <int TYPE>
bool
PosixSpinlock<TYPE>::is_shared() const
{
    return TYPE == PTHREAD_PROCESS_SHARED;
}

template <int TYPE>
const tpl_spinlock_t&
PosixSpinlock<TYPE>::spinlock_handle() const
{
    return _spinlock;
}

////////////////////////////////////////////////////////////
//              PosixMutexAttributes                      //
////////////////////////////////////////////////////////////

#if defined(__USE_GNU)
const PosixMutexAttributes PosixMutexAttributes::NORMAL(PTHREAD_MUTEX_FAST_NP);
const PosixMutexAttributes PosixMutexAttributes::RECURSIVE(PTHREAD_MUTEX_RECURSIVE_NP);
const PosixMutexAttributes PosixMutexAttributes::ERRORCHECKING(PTHREAD_MUTEX_ERRORCHECK_NP);
#elif defined(__USE_UNIX98)
const PosixMutexAttributes PosixMutexAttributes::NORMAL(PTHREAD_MUTEX_NORMAL);
const PosixMutexAttributes PosixMutexAttributes::RECURSIVE(PTHREAD_MUTEX_RECURSIVE);
const PosixMutexAttributes PosixMutexAttributes::ERRORCHECKING(PTHREAD_MUTEX_ERRORCHECK);
#endif  // __USE_UNIX98

PosixMutexAttributes::PosixMutexAttributes(int kind, int shared)
{
    TPL_VERIFY(::pthread_mutexattr_init(&_attr) == 0);

    TPL_VERIFY(::pthread_mutexattr_setpshared(&_attr, shared) == 0);

#ifdef __USE_UNIX98
    TPL_VERIFY(::pthread_mutexattr_settype(&_attr, kind) == 0);
#endif // __USE_UNIX98
}

PosixMutexAttributes::~PosixMutexAttributes()
{
    TPL_VERIFY(::pthread_mutexattr_destroy(&_attr) == 0);
}

/**
 * @return PTHREAD_MUTEX_NORMAL, PTHREAD_MUTEX_RECURSIVE,
 * PTHREAD_MUTEX_ERRORCHECK or PTHREAD_MUTEX_DEFAULT values of the kind attribute.
 */
int
PosixMutexAttributes::kind() const
{
    int k = PTHREAD_MUTEX_DEFAULT;
#ifdef __USE_UNIX98
    TPL_VERIFY(::pthread_mutexattr_gettype(&_attr, &k) == 0);
#endif // __USE_UNIX98
    return k;
}

/**
 * @return PTHREAD_PROCESS_PRIVATE or PTHREAD_PROCESS_SHARED
 * values of the sharing attribute.
 */
int
PosixMutexAttributes::shared() const
{
    int s = PTHREAD_PROCESS_PRIVATE;
    TPL_VERIFY(::pthread_mutexattr_getpshared(&_attr, &s) == 0);
    return s;
}

const pthread_mutexattr_t&
PosixMutexAttributes::mutexattr_handle() const
{
    return _attr;
}

////////////////////////////////////////////////////////////
//                      PosixMutex                        //
////////////////////////////////////////////////////////////

template <const PosixMutexAttributes& ATTRIBUTES>
PosixMutex<ATTRIBUTES>::PosixMutex()
#ifdef TPL_DEBUG
 : _host_thread(0)
#endif
{
    TPL_VERIFY(::pthread_mutex_init(&_mutex, &(ATTRIBUTES.mutexattr_handle())) == 0);
}

template <const PosixMutexAttributes& ATTRIBUTES>
PosixMutex<ATTRIBUTES>::~PosixMutex()
{
    TPL_VERIFY(::pthread_mutex_destroy(&_mutex) == 0);
}

template <const PosixMutexAttributes& ATTRIBUTES>
bool
PosixMutex<ATTRIBUTES>::try_lock() const
{
    int err = ::pthread_mutex_trylock(&_mutex);

    if (err == 0) {
#ifdef TPL_DEBUG
        _host_thread = ::pthread_self();
#endif // TPL_DEBUG
    	return true;
    }

    TPL_ASSERT(err == EBUSY);
    return false;
}

template <const PosixMutexAttributes& ATTRIBUTES>
void
PosixMutex<ATTRIBUTES>::lock() const
{
    int err;
    if ((err = ::pthread_mutex_lock(&_mutex)) != 0) {
    	TPL_ASSERT(err == EDEADLK);

// TODO: change direct output to cerr and call to abort to some
// centralized termination function
#ifdef __USE_UNIX98
        if (err == EDEADLK) {
            std::cerr << "PosixMutex: Deadlock detected" << std::endl;
            abort();
        }
#endif // __USE_UNIX98
    }

#ifdef TPL_DEBUG
    _host_thread = ::pthread_self();
#endif // TPL_DEBUG
}

template <const PosixMutexAttributes& ATTRIBUTES>
void
PosixMutex<ATTRIBUTES>::unlock() const
{
#ifdef TPL_DEBUG
    _host_thread = 0;
#endif // TPL_DEBUG

    TPL_VERIFY(::pthread_mutex_unlock(&_mutex) == 0);
}

template <const PosixMutexAttributes& ATTRIBUTES>
int
PosixMutex<ATTRIBUTES>::kind() const
{
    return ATTRIBUTES.kind();
}

template <const PosixMutexAttributes& ATTRIBUTES>
bool
PosixMutex<ATTRIBUTES>::is_shared() const
{
    return ATTRIBUTES.shared() == PTHREAD_PROCESS_SHARED;
}

template <const PosixMutexAttributes& ATTRIBUTES>
const pthread_mutex_t&
PosixMutex<ATTRIBUTES>::mutex_handle() const
{
    return _mutex;
}

#ifdef TPL_DEBUG
template <const PosixMutexAttributes& ATTRIBUTES>
pthread_t
PosixMutex<ATTRIBUTES>::owner_thread()
{
    return _owner_thread;
}
#endif // TPL_DEBUG

//------------------------------------------------------------
// Template Instantiations

template class PosixSpinlock<>;
template class PosixMutex<PosixMutexAttributes::NORMAL>;
template class PosixMutex<PosixMutexAttributes::RECURSIVE>;
template class PosixMutex<PosixMutexAttributes::ERRORCHECKING>;
