// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_POSIX_CONDITION_HH__
#define __TPL_POSIX_CONDITION_HH__

#include "libtpl/tpl_decls.hh"
#include "posix_lock_objects.hh"

#include <pthread.h>
#include <sys/time.h>

namespace tpl {

struct PolicyConditionStateless
{
    void set_signalled(bool) const {}

    bool is_signalled() const { return false; }
};

struct PolicyConditionStatefull
{
    PolicyConditionStatefull() : _signalled(false) {}

    void set_signalled(bool s) const { _signalled = s; }

    bool is_signalled() const { return _signalled; }

private:
    mutable bool _signalled;
};

typedef PolicyConditionStateless CONDITION_STATELESS;
typedef PolicyConditionStatefull CONDITION_STATEFULL;

/**
 *  @short POSIX condition attributes wrapper class.
 */
class PosixConditionAttributes
{
public:
    PosixConditionAttributes(int shared = PTHREAD_PROCESS_PRIVATE);

    ~PosixConditionAttributes();

    TPL_NO_COPY(PosixConditionAttributes);

public:  // State

    /**
     * @return PTHREAD_PROCESS_PRIVATE or PTHREAD_PROCESS_SHARED
     * values of the sharing attribute.
     */
    int shared() const;

    /**
     *	@return reference to the internal representation of mutexattr.
     */
    const pthread_condattr_t& condattr_handle() const;
	
public:  // Constants

    static const PosixConditionAttributes PRIVATE;
    static const PosixConditionAttributes SHARED;

private:
    pthread_condattr_t _attr;
};

/**
 * @short POSIX condition facility wrapper class
 *
 */
template <class POLICY, const PosixConditionAttributes& ATTRIBUTES = PosixConditionAttributes::PRIVATE>
class PosixCondition : private POLICY
{
public: // Create/Copy/Destroy

    template <const PosixMutexAttributes& A>
    PosixCondition(const PosixMutex<A>& mutex) : _mutex(mutex.mutex_handle()) {
    	init();
    }

    ~PosixCondition() {
    	destroy();
    }

    TPL_NO_COPY_TEMPLATE(PosixCondition, POLICY, ATTRIBUTES);

public:  // Usage

    /**
     *	Blocks on a condition until specified moment in time (absolute time).
     *  With statefull policy if condition has been signalled before this call
     *  it will exit immediately.
     *  XXX: if time specified in the past this call will block until signalled.
     *  @return 0 if condition signalled, otherwise - POSIX error code:
     *  ETIMEDOUT - specified time reached; EINTR - call was interrupted.
     */
    int wait_until(const struct timespec& abstime) const;

    /**
     *	Blocks on a condition for an interval in milliseconds.
     *  With statefull policy if condition has been signalled before this call
     *  it will exit immediately.
     *  XXX: if specified interval is 0 this call will block until signalled.
     *  @return 0 if condition signalled, otherwise - POSIX error code:
     *  ETIMEDOUT - specified time reached; EINTR - call was interrupted.
     */
    int wait(unsigned time_ms = 0) const;

    /**
     *	Signals the thread which is waiting on a condition.
     */
    void signal() const;

    /**
     *	Signals to all threads waiting on a condition (bradcast)
     */
    void broadcast() const;

public:  // State

    /**
     *  @return true if the condition is shared.
     */
    bool is_shared() const;

private:
    void init();
    void destroy();

    mutable pthread_cond_t _cond;	// Condition variable
    const pthread_mutex_t& _mutex;	// Mutex associated with the condition
};

};  // namespace tpl

#endif // __TPL_POSIX_CONDITION_HH__
