// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#include "posix_condition.hh"
#include "libtpl/debug.hh"

#include <errno.h>

using namespace tpl;

////////////////////////////////////////////////////////////
//		PosixConditionAttributes                  //
////////////////////////////////////////////////////////////

const PosixConditionAttributes PosixConditionAttributes::PRIVATE(PTHREAD_PROCESS_PRIVATE);
const PosixConditionAttributes PosixConditionAttributes::SHARED(PTHREAD_PROCESS_SHARED);

PosixConditionAttributes::PosixConditionAttributes(int shared)
{
    TPL_VERIFY(::pthread_condattr_init(&_attr) == 0);

    TPL_VERIFY(::pthread_condattr_setpshared(&_attr, shared) == 0);
}

PosixConditionAttributes::~PosixConditionAttributes()
{
    TPL_VERIFY(::pthread_condattr_destroy(&_attr) == 0);
}

/**
 * @return PTHREAD_PROCESS_PRIVATE or PTHREAD_PROCESS_SHARED
 * values of the sharing attribute.
 */
int
PosixConditionAttributes::shared() const
{
    int s = PTHREAD_PROCESS_PRIVATE;
    TPL_VERIFY(::pthread_condattr_getpshared(&_attr, &s) == 0);
    return s;
}

const pthread_condattr_t&
PosixConditionAttributes::condattr_handle() const
{
    return _attr;
}

////////////////////////////////////////////////////////////
//                     PosixCondition                     //
////////////////////////////////////////////////////////////

template <class POLICY, const PosixConditionAttributes& ATTRIBUTES>
void
PosixCondition<POLICY, ATTRIBUTES>::init()
{
    TPL_VERIFY(::pthread_cond_init(&_cond, &(ATTRIBUTES.condattr_handle())) == 0);
}

template <class POLICY, const PosixConditionAttributes& ATTRIBUTES>
void
PosixCondition<POLICY, ATTRIBUTES>::destroy()
{
    TPL_VERIFY(::pthread_cond_destroy(&_cond) == 0);
}

template <class POLICY, const PosixConditionAttributes& ATTRIBUTES>
int
PosixCondition<POLICY, ATTRIBUTES>::wait_until(const struct timespec& abstime) const
{
    if (POLICY::is_signalled()) {
    	POLICY::set_signalled(false);
    	return 0;
    }

    int err = ::pthread_cond_timedwait(&_cond, const_cast<pthread_mutex_t*>(&_mutex), &abstime);
    TPL_VERIFY(err == 0 || err == ETIMEDOUT || err == EINTR);
    POLICY::set_signalled(false);

    return err;
}

template <class POLICY, const PosixConditionAttributes& ATTRIBUTES>
int
PosixCondition<POLICY, ATTRIBUTES>::wait(unsigned time_ms) const
{
    if (POLICY::is_signalled()) {
    	POLICY::set_signalled(false);
    	return 0;
    }

    int err = 0;
    struct timespec timeout;
    struct timeval now;

    if (time_ms == 0) {
    	err = ::pthread_cond_wait(&_cond, const_cast<pthread_mutex_t*>(&_mutex));
    	TPL_VERIFY(err == 0 || err == EINTR);
    }
    else {
    	gettimeofday(&now, NULL);
    	timeout.tv_nsec = (time_ms % 1000) * 1000 + now.tv_usec;
    	timeout.tv_sec =  (time_ms / 1000) + now.tv_sec + (timeout.tv_nsec / 1000000);
    	timeout.tv_nsec = (timeout.tv_nsec % 1000000) * 1000;

    	err = ::pthread_cond_timedwait(&_cond, const_cast<pthread_mutex_t*>(&_mutex), &timeout);
    	TPL_VERIFY(err == 0 || err == ETIMEDOUT || err == EINTR);
    }
    POLICY::set_signalled(false);

    return err;
}

template <class POLICY, const PosixConditionAttributes& ATTRIBUTES>
void
PosixCondition<POLICY, ATTRIBUTES>::signal() const
{
    POLICY::set_signalled(true);
    TPL_VERIFY(::pthread_cond_signal(&_cond) == 0);
}

template <class POLICY, const PosixConditionAttributes& ATTRIBUTES>
void
PosixCondition<POLICY, ATTRIBUTES>::broadcast() const
{
    POLICY::set_signalled(true);
    TPL_VERIFY(::pthread_cond_broadcast(&_cond) == 0);
}

template <class POLICY, const PosixConditionAttributes& ATTRIBUTES>
bool
PosixCondition<POLICY, ATTRIBUTES>::is_shared() const
{
    return ATTRIBUTES.shared() == PTHREAD_PROCESS_SHARED;
}

//------------------------------------------------------------
// Template Instantiations

template class PosixCondition<CONDITION_STATELESS>;
template class PosixCondition<CONDITION_STATEFULL>;
