// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_ATOMIC_HH__
#define __TPL_ATOMIC_HH__

#include <stddef.h>

namespace tpl {

#ifndef atomic_t
typedef struct { volatile int counter; } atomic_t;
#endif

//  Machine dependent atomic operations

/**
 *  Performs safe atomic increment of integer value
 *  @return last value of integer before increment
 */
inline unsigned long
atomic_inc(volatile unsigned long &ref)
{
#if defined __i486__ || defined __i586__ || defined __i686__
    size_t __val;
    asm ("lock; xaddl %%eax, (%%edx)"
	 : "=a" (__val)
	 : "0" (1), "m" (ref), "d" (&ref)
	 : "memory");
    return __val;
#else
    return ref++;
#endif
}

/**
 *  Performs safe decrement of integer value
 *  @return last value of integer before decrement
 */
inline unsigned long
atomic_dec(volatile unsigned long &ref)
{
#if defined __i486__ || defined __i586__ || defined __i686__
    size_t __val;
    asm ("lock; xaddl %%eax, (%%edx)"
	 : "=a" (__val)
	 : "0" (-1), "m" (ref), "d" (&ref)
	 : "memory");
    return __val;
#else
    return ref--;
#endif
}

/**
 *  Performs safe atomic increment of integer value
 *  @return last value of integer before increment
 */
inline unsigned short
atomic_inc (volatile unsigned short &ref)
{
#if defined __i486__ || defined __i586__ || defined __i686__
    size_t __val;
    asm ("lock; xaddl %%ax, (%%dx)"
	 : "=a" (__val)
	 : "0" (1), "m" (ref), "d" (&ref)
	 : "memory");
    return __val;
#else
    return ref++;
#endif
}

/**
 *  Performs safe decrement of integer value
 *  @return last value of integer before decrement
 */
inline unsigned short
atomic_dec (volatile unsigned short &ref)
{
#if defined __i486__ || defined __i586__ || defined __i686__
    size_t __val;
    asm ("lock; xaddl %%ax, (%%dx)"
	 : "=a" (__val)
	 : "0" (-1), "m" (ref), "d" (&ref)
	 : "memory");
    return __val;
#else
    return ref--;
#endif
}

};  // namespace tpl

#endif //__TPL_ATOMIC_HH__
