// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_QUEUE_PANEL_HH__
#define __TPL_QUEUE_PANEL_HH__

#include "libtpl/tpl_decls.hh"
#include "blocking_queue.hh"
#include "libtpl/thread/condition.hh"
#include "libtpl/thread/lock.hh"

#include <map>
#include <list>

namespace tpl {

template <class T>
class LinkedBlockingQueue;

/**
 *  @short Queue Panel.
 */
template <class T>
class QueuePanel
{
protected:  // Create/Copy/Destroy

    /**
     *  Create default queue panel with shared mutex.
     */
    QueuePanel(Mutex& mutex);

    /**
     *  Destroyes queue.
     *  XXX: Message objects are destroyed and all queues are cleared
     */
    virtual ~QueuePanel();

    TPL_NO_COPY_TEMPLATE(QueuePanel, T);

protected:  // Non-locking interface

    /**
     *  Create a queue (no lock).
     *  XXX: returns an id of newly created queue.
     *  @param max sets maximum size of created queue
     */
    int create_queue_nl(size_t max = 0);

    /**
     *  Removes queue specified by id (no lock).
     *  XXX: removes queue from the panel and deletes all elements in the queue.
     */
    void remove_queue_nl(int id);

    /**
     *  Clears the queue specified by id (no lock).
     */
    void clear_nl(int id);

    /**
     *  Reset queue after it has been stalled (no lock).
     */
    void reset_nl(int id);

    /**
     *  Activate queue after it has been deactivated (no lock).
     */
    void activate_nl(int id);

    /**
     *  Set inactive mode and release all waiting putters and getters (no lock).
     *  XXX: Call blocks until all putting and getting threads released.
     */
    void deactivate_nl(int id);

    /**
     *  Returns number of registered queues (no lock).
     */
    size_t queue_count_nl() const;

protected:  // Usage

    /**
     *  Put element into the queue specified by id (no lock).
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is saturated the call will block
     *  and contend for the first available slot in the queue after
     *  get() operation is called.
     *  @return true if element has been inserted into the queue.
     */
    bool put_nl(int id, T* message, unsigned timeout = 0);

    /**
     *  Try to put element into the queue without blocking (no lock).
     *  XXX: The call will return immediately if the queue is saturated.
     *  @return true if element has been inserted into the queue.
     */
    bool try_put_nl(int id, T* element);

    /**
     *  Put multiple elements in bulk mode into the queue specified by id (no lock).
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is saturated the call will block
     *  and contend for the elements.size() available slots in the queue after
     *  get() operation is called.
     *  @return true if all elements have been inserted into the queue.
     */
    bool put_bulk_nl(int id, std::list<T*> elements, unsigned timeout = 0);

    /**
     *  Try to put multiple elements in bulk mode into the queue specified by id (no lock).
     *  XXX: The call will return immediately if the queue is saturated.
     *  @return true if all elements have been inserted into the queue.
     */
    bool try_put_bulk_nl(int id, std::list<T*> elements);

    /**
     *  Get element from the queue specified by id (no lock).
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is empty the call will block
     *  and contend for the first available element in the queue after
     *  put() operation is called.
     *  @return pointer to the element withdrawn from the queue
     *  or NULL if the queue was empty.
     */
    T* get_nl(int id, unsigned timeout = 0);

    /**
     *  Try to get an element from the queue specified by id (no lock).
     *  XXX: The call will return immediately if the queue is empty.
     *  @return pointer to the element withdrawn from the queue
     *  or NULL if the queue was empty.
     */
    T* try_get_nl(int id);

    /**
     *  Get multiple elements from the queue in bulk mode (no lock).
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is empty the call will block
     *  and contend for the first bulk_size available elements in the queue after
     *  put() operation is called.
     *  @return a list of pointers to the elements withdrawn from the queue
     *  or an empty list if the queue was empty.
     */
    std::list<T*> get_bulk_nl(int id, size_t bulk_size, unsigned timeout = 0);

    /**
     *  Try to get multiple elements from the queue in bulk mode without blocking.
     *  XXX: The call will return immediately if the queue is empty.
     *  @return a list of pointers to the elements withdrawn from the queue
     *  or an empty list if the queue was empty.
     */
    std::list<T*> try_get_bulk_nl(int id, size_t bulk_size);

protected:  // Properties

    /**
     *  Support for max_queue_size property on per queue basis.
     */
    void set_max_elements_nl(int id, size_t max);

    size_t max_elements_nl(int id) const;

protected:  // State

    /**
     *  Returns true when queue is activated.
     */
    bool is_active_nl(int id) const;

    /**
     *  Returns whether the queue specified by id is empty.
     */
    bool is_empty_nl(int id) const;

    /**
     *  Returns number of messages in the queue specified by id.
     */
    size_t size_nl(int id) const;

private:  // Attributes

    /** Sequential queue id generator */
    int _current_id;
    /** Internal collection of queues */
    std::map<int, LinkedBlockingQueue<T>*> _queues;

    /**
     *  Shared mutex - used to synchronize get/put in all queues
     *  also used for create/remove operations in queue panel.
     */
    Mutex& _mutex;
};

};  // namespace tpl

#ifndef NO_COMPILE_INSTANTIATE
#include "queue_panel.cc"
#endif

#endif // __TPL_QUEUE_PANEL_HH__
