// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#include "linked_queue.hh"
#include "libtpl/thread/lock_guards.hh"
#include "libtpl/debug.hh"

#include <iostream>

using namespace tpl;

////////////////////////////////////////////////////////////
//                     QueuePanel<T>                      //
////////////////////////////////////////////////////////////

template <class T>
QueuePanel<T>::QueuePanel(Mutex& mutex) : _current_id(0), _mutex(mutex)
{}

template <class T>
QueuePanel<T>::~QueuePanel()
{
    typename std::map<int, LinkedBlockingQueue<T>*>::iterator beg = _queues.begin();
    while (beg != _queues.end()) {
    	LinkedBlockingQueue<T> *q = beg->second;
    	_queues.erase(beg++);
    	delete q;
    }
}

template <class T>
bool
QueuePanel<T>::put_nl(int id, T* element, unsigned timeout)
{
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	return q->put_nl(element, timeout);
    }
    return false;
}

template <class T>
bool
QueuePanel<T>::try_put_nl(int id, T* element)
{
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	return q->try_put_nl(element);
    }
    return false;
}

template <class T>
bool
QueuePanel<T>::put_bulk_nl(int id, std::list<T*> elements, unsigned timeout)
{
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	return q->put_bulk_nl(elements, timeout);
    }
    return false;
}

template <class T>
bool
QueuePanel<T>::try_put_bulk_nl(int id, std::list<T*> elements)
{
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	return q->try_put_bulk_nl(elements);
    }
    return false;
}

template <class T>
T*
QueuePanel<T>::get_nl(int id, unsigned timeout)
{
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	return q->get_nl(timeout);
    }
    return 0;
}

template <class T>
T*
QueuePanel<T>::try_get_nl(int id)
{
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	return q->try_get_nl();
    }
    return 0;
}

template <class T>
std::list<T*>
QueuePanel<T>::get_bulk_nl(int id, size_t bulk_size, unsigned timeout)
{
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	return q->get_bulk_nl(bulk_size, timeout);
    }
    return std::list<T*>();
}

template <class T>
std::list<T*>
QueuePanel<T>::try_get_bulk_nl(int id, size_t bulk_size)
{
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	return q->try_get_bulk_nl(bulk_size);
    }
    return std::list<T*>();
}

template <class T>
int
QueuePanel<T>::create_queue_nl(size_t max)
{
    LinkedBlockingQueue<T> *q = new LinkedBlockingQueue<T>(&_mutex, max);
    // search for empty slot
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi;
    do {
    	qi = _queues.find(++_current_id);
    } while (qi != _queues.end() && _current_id);
    TPL_ASSERT(_current_id);
    _queues[_current_id] = q;
    return _current_id;
}

template <class T>
void
QueuePanel<T>::remove_queue_nl(int id)
{
    typename std::map<int, LinkedBlockingQueue<T>*>::iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	_queues.erase(qi);
    	delete q;
    }
}

template <class T>
void
QueuePanel<T>::clear_nl(int id)
{
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	q->clear_nl();
    }
}

template <class T>
void
QueuePanel<T>::reset_nl(int id)
{
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	q->reset_nl();
    }
}

template <class T>
void
QueuePanel<T>::activate_nl(int id)
{	
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	q->activate();
    }
}

template <class T>
void
QueuePanel<T>::deactivate_nl(int id)
{	
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	q->deactivate_nl();
    }
}

template <class T>
size_t
QueuePanel<T>::queue_count_nl() const
{
    return _queues.size();
}

template <class T>
void
QueuePanel<T>::set_max_elements_nl(int id, size_t max)
{
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	q->set_max_elements(max);
    }
}

template <class T>
size_t
QueuePanel<T>::max_elements_nl(int id) const
{
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	return q->max_elements();
    }
    return 0;
}

template <class T>
bool
QueuePanel<T>::is_active_nl(int id) const
{
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	return q->is_active();
    }
    return false;
}

template <class T>
bool
QueuePanel<T>::is_empty_nl(int id) const
{
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	return q->is_empty_nl();
    }
    return true;
}

template <class T>
size_t
QueuePanel<T>::size_nl(int id) const
{
    typename std::map<int, LinkedBlockingQueue<T>*>::const_iterator qi = _queues.find(id);
    if (qi != _queues.end()) {
    	LinkedBlockingQueue<T> *q = qi->second;
    	return q->size_nl();
    }
    return 0;
}

