// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_PRIORITY_QUEUE_HH__
#define __TPL_PRIORITY_QUEUE_HH__

#include "libtpl/tpl_decls.hh"
#include "queue_panel.hh"
#include "libtpl/thread/condition.hh"
#include "libtpl/thread/lock.hh"

#include <list>
#include <vector>

namespace tpl {

template <class T>
class PriorityQueue;

template <class T>
inline std::ostream& operator<< (std::ostream& os, const PriorityQueue<T>& obj)
{
    obj.print_on(os);
    return os;
}

/**
 *  @short MultiQueue.
 */
template <class T>
class PriorityQueue : public QueuePanel<T>
{
public:  // Create/Copy/Destroy

    /**
     *  Create a priority queue.
     */
    PriorityQueue();

    /**
     *  Destroy priority queue.
     *  XXX: Message objects are destroyed and queue is cleared
     */
    virtual ~PriorityQueue();

    TPL_NO_COPY_TEMPLATE(PriorityQueue, T);

public:  // Usage

    /**
     *  Put element into the queue specified by priority level.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is saturated the call will block
     *  and contend for the first available slot in the queue after
     *  get() operation is called.
     *  @return true if element has been inserted into the queue.
     */
    virtual bool put(int priority, T* element, unsigned timeout = 0);

    /**
     *  Try to put element into the queue without blocking.
     *  XXX: The call will return immediately if the queue is saturated.
     *  @return true if element has been inserted into the queue.
     */
    virtual bool try_put(int priority, T* element);

    /**
     *  Put multiple elements in bulk mode into the queue specified by priority level.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is saturated the call will block
     *  and contend for the elements.size() available slots in the queue after
     *  get() operation is called.
     *  @return true if all elements have been inserted into the queue.
     */
    virtual bool put_bulk(int priority, std::list<T*> elements, unsigned timeout = 0);

    /**
     *  Try to put multiple elements in bulk mode into the queue without blocking.
     *  XXX: The call will return immediately if the queue is saturated.
     *  @return true if all elements have been inserted into the queue.
     */
    virtual bool try_put_bulk(int priority, std::list<T*> elements);

    /**
     *  Get next element from the queue.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is empty the call will block
     *  and contend for the first available element in the queue after
     *  put() operation is called.
     *  @return pointer to the element withdrawn from the queue
     *  or NULL if the queue was empty.
     */
    virtual T* get(unsigned timeout = 0);

    /**
     *  Try to get an element from the queue.
     *  XXX: The call will return immediately if the queue is empty.
     *  @return pointer to the element withdrawn from the queue
     *  or NULL if the queue was empty.
     */
    virtual T* try_get();

public:  // Management

    /**
     *  Create a queue of specified length for specified priority level.
     */
    void create_queue(int priority, size_t max = 0);

    /**
     *  Remove a queue specified by priority level.
     */
    void remove_queue(int priority);

    /**
     *  Clear all queues registered with priority queue.
     */
    void clear_all();

    /**
     *  Clear the queue specified by priority level.
     */
    void clear(int priority);

    /**
     *  Reset all queues registered with priority queue.
     */
    void reset();

    /**
     *  Activate all queues registered with priority queue.
     */
    void activate();

    /**
     *  Set inactive mode and release all waiting putters and getters for all queues in priority queue.
     *  XXX: Call blocks until all putting and getting threads are released.
     */
    void deactivate();

public:  // Properties

    size_t max_elements(int priority) const;

    void set_max_elements(int priority, size_t max);

public:  // State

    /**
     *  @return true when all queues in multi-queue are activated.
     */
    bool is_active() const;

    /**
     *  @return true if all queues are empty.
     */
    bool is_empty() const;

    /**
     *  @return number of messages in all queues.
     */
    size_t size() const;

    /**
     *  @return number of messages in the queue specified by priority level.
     */
    size_t size(int priority) const;

    /**
     *  @return highest priority level that has non-empty queue.
     */
    int ready_priority() const;

    /**
     *  @return number of registered queues.
     */
    size_t queue_count() const;

#ifdef TPL_DEBUG

    /**
     *  @return number of messages in all queues (non-locking).
     */
    size_t size_debug() const;

    /**
     *  @return number of messages in the queue specified by priority level (non-locking).
     */
    size_t size_debug(int priority) const;

    /**
     *  @return number of registered queues (non-locking).
     */
    size_t queues_count_debug() const;

#endif // TPL_DEBUG

private:

    using QueuePanel<T>::size_nl;

    size_t size_nl() const;

    T* priority_get(unsigned timeout);

    T* priority_try_get();

public:  // Printing

    friend std::ostream& operator<< <T>(std::ostream& os, const PriorityQueue<T>& obj);

protected:

    virtual void print_on(std::ostream& os) const;

private:  // Attributes

    std::map<int, int, std::less<int> > _id_map;

    // mutex for create/remove sync. and get.
    mutable Mutex _mutex;
    Condition _get_cond;

    bool _shutdown; 	// shutdown flag
    size_t _get_waiting; 	// waiting queue size for get.
};

};  // namespace tpl

#ifndef NO_COMPILE_INSTANTIATE
#include "priority_queue.cc"
#endif

#endif // __TPL_MULTI_QUEUE_HH__
