// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#include "libtpl/thread/lock_guards.hh"
#include "libtpl/debug.hh"

using namespace tpl;

////////////////////////////////////////////////////////////
//                      PriorityQueue<T>                  //
////////////////////////////////////////////////////////////

template <class T>
PriorityQueue<T>::PriorityQueue() : QueuePanel<T>(_mutex),
   _get_cond(_mutex), _shutdown(false), _get_waiting(0)
{
}

template <class T>
PriorityQueue<T>::~PriorityQueue()
{
}

template <class T>
bool
PriorityQueue<T>::put(int priority, T* element, unsigned timeout)
{
    TPL_ASSERT(element);

    LockGuard<Mutex> guard(_mutex);
    int id = _id_map[priority];
    if (!_shutdown  && put_nl(id, element, timeout)) {
    	_get_cond.signal();
    	return true;
    }
    return false;
}

template <class T>
bool
PriorityQueue<T>::try_put(int priority, T* element)
{
    TPL_ASSERT(element);

    LockGuard<Mutex> guard(_mutex);
    int id = _id_map[priority];
    if (!_shutdown && try_put_nl(id, element)) {
    	_get_cond.signal();
    	return true;
    }
    return false;
}

template <class T>
bool
PriorityQueue<T>::put_bulk(int priority, std::list<T*> elements, unsigned timeout)
{
    TPL_ASSERT(!elements.empty());

    LockGuard<Mutex> guard(_mutex);
    int id = _id_map[priority];
    if (!_shutdown && put_bulk_nl(id, elements, timeout)) {
    	_get_cond.signal();
    	return true;
    }
    return false;
}

template <class T>
bool
PriorityQueue<T>::try_put_bulk(int priority, std::list<T*> elements)
{
    TPL_ASSERT(!elements.empty());

    LockGuard<Mutex> guard(_mutex);
    int id = _id_map[priority];
    if (!_shutdown && try_put_bulk_nl(id, elements)) {
    	_get_cond.signal();
    	return true;
    }
    return false;
}

template <class T>
T*
PriorityQueue<T>::get(unsigned timeout)
{
    LockGuard<Mutex> guard(_mutex);
    if (!_shutdown) {
    	return priority_get(timeout);
    }
    return 0;
}

template <class T>
T*
PriorityQueue<T>::try_get()
{
    LockGuard<Mutex> guard(_mutex);
    if (!_shutdown) {
    	return priority_try_get();
    }
    return 0;
}

template <class T>
void
PriorityQueue<T>::create_queue(int priority, size_t max)
{
    LockGuard<Mutex> guard(_mutex);
    int id = this->create_queue_nl(max);
    _id_map[priority] = id;
}

template <class T>
void
PriorityQueue<T>::remove_queue(int priority)
{
    TPL_ASSERT(id != 0);

    LockGuard<Mutex> guard(_mutex);
    std::map<int, int>::const_iterator it = _id_map.find(priority);
    this->remove_queue_nl(it->second);
    _id_map.erase(priority);
}

template <class T>
void
PriorityQueue<T>::reset()
{
    LockGuard<Mutex> guard(_mutex);

    std::map<int, int>::const_iterator beg = _id_map.begin();
    while (beg != _id_map.end()) {
        this->reset_nl(beg->second);
        ++beg;
    }
}

template <class T>
void
PriorityQueue<T>::activate()
{
    LockGuard<Mutex> guard(_mutex);

    std::map<int, int>::const_iterator beg = _id_map.begin();
    while (beg != _id_map.end()) {
        this->activate_nl(beg->second);
        ++beg;
    }
    _shutdown = false;
}

template <class T>
void
PriorityQueue<T>::deactivate()
{
    LockGuard<Mutex> guard(_mutex);
    _shutdown = true;

    std::map<int, int>::const_iterator beg = _id_map.begin();
    while (beg != _id_map.end()) {
    	this->deactivate_nl(beg->second);
    	++beg;
    }

    while(_get_waiting) {
    	_get_cond.broadcast();
    	_get_cond.wait(100);
    }
}

template <class T>
void
PriorityQueue<T>::clear_all()
{
    LockGuard<Mutex> guard(_mutex);

    std::map<int, int>::const_iterator beg = _id_map.begin();
    while (beg != _id_map.end()) {
    	this->clear_nl(beg->second);
    	++beg;
    }
}

template <class T>
void
PriorityQueue<T>::clear(int priority)
{
    TPL_ASSERT(id != 0);

    LockGuard<Mutex> guard(_mutex);
    std::map<int, int>::const_iterator it = _id_map.find(priority);
    this->clear_nl(it->second);
}

template <class T>
size_t
PriorityQueue<T>::max_elements(int priority) const
{
    TPL_ASSERT(id != 0);

    LockGuard<Mutex> guard(_mutex);
    std::map<int, int>::const_iterator it = _id_map.find(priority);
    return this->max_elements_nl(it->second);
}

template <class T>
void
PriorityQueue<T>::set_max_elements(int priority, size_t max)
{
    TPL_ASSERT(id != 0);

    LockGuard<Mutex> guard(_mutex);
    std::map<int, int>::const_iterator it = _id_map.find(priority);
    this->set_max_elements_nl(it->second, max);
}

template <class T>
bool
PriorityQueue<T>::is_active() const
{
    return !_shutdown;
}

template <class T>
bool
PriorityQueue<T>::is_empty() const
{
    LockGuard<Mutex> guard(_mutex);
    std::map<int, int>::const_iterator beg = _id_map.begin();
    while (beg != _id_map.end()) {
        if (!this->is_empty_nl(beg->second)) return false;
        ++beg;
    }
    return true;
}

template <class T>
size_t
PriorityQueue<T>::size() const
{
    LockGuard<Mutex> guard(_mutex);
    return this->size_nl();
}

template <class T>
size_t
PriorityQueue<T>::size(int priority) const
{
    LockGuard<Mutex> guard(_mutex);
    std::map<int, int>::const_iterator it = _id_map.find(priority);
    return this->size_nl(it->second);
}

template <class T>
size_t
PriorityQueue<T>::queue_count() const
{
    LockGuard<Mutex> guard(_mutex);
    return this->queue_count_nl();
}

template <class T>
int
PriorityQueue<T>::ready_priority() const
{
    LockGuard<Mutex> guard(_mutex);
    std::map<int, int>::const_iterator beg = _id_map.begin();
    while (beg != _id_map.end()) {
        if (!this->is_empty_nl(beg->second)) return beg->first;
        ++beg;
    }
    return -1;
}

template <class T>
size_t
PriorityQueue<T>::size_nl() const
{
    size_t sz = 0;
    std::map<int, int>::const_iterator beg = _id_map.begin();
    while (beg != _id_map.end()) {
        sz += this->size_nl(beg->second);
        ++beg;
    }
    return sz;
}

#ifdef TPL_DEBUG

template <class T>
int
PriorityQueue<T>::size_debug() const
{
    return this->size_nl();
}

template <class T>
int
PriorityQueue<T>::size_debug(int priority) const
{
    return this->size_nl(priority);
}

template <class T>
int
PriorityQueue<T>::queues_count_debug() const
{
    return this->queues_count_nl();
}

#endif // TPL_DEBUG

template <class T>
void
PriorityQueue<T>::print_on(std::ostream&) const
{}

template <class T>
T*
PriorityQueue<T>::priority_get(unsigned timeout)
{
    bool is_expired = false;
    if (_id_map.size() == 0) {
    	_get_waiting++;
    	_get_cond.wait(timeout);
    	_get_waiting--;
    	is_expired = true;
    }

    while (!_shutdown && _id_map.size() > 0) {
    	std::map<int, int>::const_iterator beg = _id_map.begin();
    	while (beg != _id_map.end()) {
    	    if (this->is_active_nl(beg->second) && !this->is_empty_nl(beg->second))
    	    	return this->try_get_nl(beg->second);
    	    ++beg;
    	}

    	if (is_expired)
    	    return 0;

    	_get_waiting++;
    	_get_cond.wait(timeout);
    	_get_waiting--;
    	is_expired = true; // protect ourself from infinite loop in get()
    }

    return 0;
}

template <class T>
T*
PriorityQueue<T>::priority_try_get()
{
    if (_id_map.size() == 0)
    	return 0;

    std::map<int, int>::const_iterator beg = _id_map.begin();
    while (beg != _id_map.end()) {
    	if (this->is_active_nl(beg->second) && !this->is_empty_nl(beg->second))
    	    return this->try_get_nl(beg->second);
    	++beg;
    }
    return 0;
}

