// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_POLICY_QUEUE_HH__
#define __TPL_POLICY_QUEUE_HH__

#include "libtpl/tpl_decls.hh"
#include "blocking_queue.hh"
#include "libtpl/thread/condition.hh"
#include "libtpl/thread/lock.hh"

#include <list>

namespace tpl {

template <class T>
class QueuePanel;

/**
 *  Blocking queue implementation based on the STL linked list.
 */
template <class T,
	  template <class> class GET_POLICY,
	  template <class> class PUT_POLICY,
	  template <class> class STORAGE >
class PolicyBlockingQueue : public BlockingQueue<T>,
			public GET_POLICY<T>,
			public PUT_POLICY<T>,
			public STORAGE<T>
{
    friend class QueuePanel<T>;

public:  // Create/Copy/Destroy

    /**
     *  Create default blocking queue.
     */
    PolicyBlockingQueue(size_t max = 0);

    /**
     *  Create blocking queue with external mutex.
     */
    PolicyBlockingQueue(Mutex* mutex, size_t max = 0);

    /**
     *  Destroyes queue.
     *  XXX: Message objects are destroyed and queue is cleard
     */
    virtual ~LinkedBlockingQueue();

    TPL_NO_COPY_TEMPLATE(LinkedBlockingQueue, T);

public:  // Usage

    /**
     *  Put element into the queue.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is saturated the call will block
     *  and contend for the first available slot in the queue after
     *  get() operation is called.
     *  @return true if element has been inserted into the queue.
     */
    virtual bool put(T* element, unsigned timeout = 0);

    /**
     *  Try to put element into the queue without blocking.
     *  XXX: The call will return immediately if the queue is saturated.
     *  @return true if element has been inserted into the queue.
     */
    virtual bool try_put(T* element);

    /**
     *  Put multiple elements into the queue in bulk mode.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is saturated the call will block
     *  and contend for the elements.size() available slots in the queue after
     *  get() operation is called.
     *  @return true if all elements have been inserted into the queue.
     */
    virtual bool put_bulk(std::list<T*> elements, unsigned timeout = 0);

    /**
     *  Try to put multiple elements into the queue in bulk mode.
     *  XXX: The call will return immediately if the queue is saturated.
     *  @return true if all elements have been inserted into the queue.
     */
    virtual bool try_put_bulk(std::list<T*> elements);

    /**
     *  Get element from the queue.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is empty the call will block
     *  and contend for the first available element in the queue after
     *  put() operation is called.
     *  @return pointer to the element withdrawn from the queue
     *  or NULL if the queue was empty.
     */
    virtual T* get(unsigned timeout = 0);

    /**
     *  Try to get an element from the queue.
     *  XXX: The call will return immediately if the queue is empty.
     *  @return pointer to the element withdrawn from the queue
     *  or NULL if the queue was empty.
     */
    virtual T* try_get();

    /**
     *  Get multiple elements from the queue in bulk mode.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is empty the call will block
     *  and contend for the first bulk_size available elements in the queue after
     *  put() operation is called.
     *  @return a list of pointers to the elements withdrawn from the queue
     *  or an empty list if the queue was empty.
     */
    virtual std::list<T*> get_bulk(size_t bulk_size, unsigned timeout = 0);

    /**
     *  Try to get multiple elements from the queue in bulk mode without blocking.
     *  XXX: The call will return immediately if the queue is empty.
     *  @return a list of pointers to the elements withdrawn from the queue
     *  or an empty list if the queue was empty.
     */
    virtual std::list<T*> try_get_bulk(size_t bulk_size);

public:  // Management

    /**
     *  Clears the queue.
     */
    virtual void clear();

    /**
     *  Activates the queue.
     */
    virtual void activate();

    /**
     *  Deactivates the queue.
     */
    virtual void deactivate();

    /**
     *  Resets the queue if stalled.
     */
    void reset();

public:  // Properties

    /**
     *  Supprt for max_queue_size property.
     */
    void set_max_elements(size_t max);

    size_t max_elements() const;

public:     //// State

    /**
     *  Returns true when queue is not in shutdown state.
     */
    virtual bool is_active() const;

    /**
     *  Returns true when queue is empty.
     */
    virtual bool is_empty() const;

    /**
     *  Returns number of messages in the queue.
     */
    virtual size_t size() const;

#ifdef TPL_DEBUG

    /**
     *  Returns true when mutex is locked.
     */
    bool is_locked() const;

    /**
     *  Returns number of messages in the queue  without locking mutex.
     */
    size_t size_debug() const;

    /**
     *  Returns thread id of host thread.
     */
    int owner_thread() const;

#endif // TPL_DEBUG

protected:  // non-blocking interface for descendants

    virtual bool put_nl(T* element, unsigned timeout = 0);

    virtual bool try_put_nl(T* element);

    virtual bool put_bulk_nl(std::list<T*> elements, unsigned timeout = 0);

    virtual bool try_put_bulk_nl(std::list<T*> elements);

    virtual T* get_nl(unsigned timeout = 0);

    virtual T* try_get_nl();

    virtual std::list<T*> get_bulk_nl(size_t bulk_size, unsigned timeout = 0);

    virtual std::list<T*> try_get_bulk_nl(size_t bulk_size);

    virtual void clear_nl();

    virtual void reset_nl();

    virtual void deactivate_nl();

    virtual bool is_empty_nl() const;

    virtual size_t size_nl() const;

protected:  // Printing

    virtual void print_on(std::ostream& os) const;

private:  // Attributes

    size_t _max_elements;	// Max queue size.
    std::list<T*> _queue;	// Linked list queue representation.

    Mutex* _mutex;		// mutex synchronizing all operations and conditions.
    Condition _get_cond;	// get operations condition
    Condition _put_cond;	// put operations condition

    bool _shutdown;		// shutdown flag
    bool _shared_mutex;		// indicates whether the mutex is private or shared
    size_t _put_waiting;	// waiting queue size for put.
    size_t _get_waiting;	// waiting queue size for get
};

};  // namespace tpl

#ifndef NO_COMPILE_INSTANTIATE
#include "policy_queue.cc"
#endif

#endif // __TPL_LINKED_QUEUE_HH__
