// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_MULTI_QUEUE_HH__
#define __TPL_MULTI_QUEUE_HH__

#include "libtpl/tpl_decls.hh"
#include "queue_panel.hh"
#include "libtpl/thread/condition.hh"
#include "libtpl/thread/lock.hh"

#include <list>
#include <vector>

namespace tpl {

template <class T>
class MultiQueue;

template <class T>
inline std::ostream& operator<< (std::ostream& os, const MultiQueue<T>& obj)
{
    obj.print_on(os);
    return os;
}

/**
 *  @short MultiQueue.
 */
template <class T>
class MultiQueue : public QueuePanel<T>
{
public:  // Create/Copy/Destroy

    /**
     *  Create a multi queue.
     */
    MultiQueue();

    /**
     *  Destroy multi queue.
     *  XXX: Message objects are destroyed and queue is cleared
     */
    virtual ~MultiQueue();

    TPL_NO_COPY_TEMPLATE(MultiQueue, T);

public:  // Usage

    /**
     *  Put element into the queue specified by id.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is saturated the call will block
     *  and contend for the first available slot in the queue after
     *  get() operation is called.
     *  @return true if element has been inserted into the queue.
     */
    virtual bool put(int id, T* element, unsigned timeout = 0);

    /**
     *  Try to put element into the queue without blocking.
     *  XXX: The call will return immediately if the queue is saturated.
     *  @return true if element has been inserted into the queue.
     */
    virtual bool try_put(int id, T* element);

    /**
     *  Put multiple elements in bulk mode into the queue specified by id.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is saturated the call will block
     *  and contend for the elements.size() available slots in the queue after
     *  get() operation is called.
     *  @return true if all elements have been inserted into the queue.
     */
    virtual bool put_bulk(int id, std::list<T*> elements, unsigned timeout = 0);

    /**
     *  Try to put multiple elements in bulk mode into the queue specified by id (no lock).
     *  XXX: The call will return immediately if the queue is saturated.
     *  @return true if all elements have been inserted into the queue.
     */
    virtual bool try_put_bulk(int id, std::list<T*> elements);

    /**
     *  Get element from the queue.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is empty the call will block
     *  and contend for the first available element in the queue after
     *  put() operation is called.
     *  @return pointer to the element withdrawn from the queue
     *  or NULL if the queue was empty.
     */
    virtual T* get(unsigned timeout = 0);

    /**
     *  Try to get an element from the queue.
     *  XXX: The call will return immediately if the queue is empty.
     *  @return pointer to the element withdrawn from the queue
     *  or NULL if the queue was empty.
     */
    virtual T* try_get();

    /**
     *  Get multiple elements from the queue in bulk mode.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is empty the call will block
     *  and contend for the first bulk_size available elements in the queue after
     *  put() operation is called.
     *  @return a list of pointers to the elements withdrawn from the queue
     *  or an empty list if the queue was empty.
     */
    virtual std::list<T*> get_bulk(size_t bulk_size, unsigned timeout = 0);

    /**
     *  Try to get multiple elements from the queue in bulk mode without blocking.
     *  XXX: The call will return immediately if the queue is empty.
     *  @return a list of pointers to the elements withdrawn from the queue
     *  or an empty list if the queue was empty.
     */
    virtual std::list<T*> try_get_bulk(size_t bulk_size);

public:  // Management

    /**
     *  Create a queue of specified length.
     *  @return unique queue-id.
     */
    virtual int create_queue(size_t max = 0);

    /**
     *  Remove a queue by id.
     */
    virtual void remove_queue(int id);

    /**
     *  Clear all queues registered with multi-queue.
     */
    virtual void clear_all();

    /**
     *  Clear the queue specified by id.
     */
    virtual void clear(int id);

    /**
     *  Reset all queues registered with multi-queue.
     */
    virtual void reset_all();

    /**
     *  Reset the queue specified by id.
     */
    virtual void reset(int id);

    /**
     *  Activate all queues registered with multi-queue.
     */
    virtual void activate_all();

    /**
     *  Activate queue after it has been deactivated.
     */
    virtual void activate(int id);

    /**
     *  Set inactive mode and release all waiting putters and getters for all queues in multi-queue.
     *  XXX: Call blocks until all putting and getting threads are released.
     */
    virtual void deactivate_all();

    /**
     *  Set inactive mode and release all waiting putters and getters.
     *  XXX: Call blocks until all putting and getting threads are released.
     */
    virtual void deactivate(int id);

public:  // Properties

    size_t max_elements(int id) const;

    void set_max_elements(int id, size_t max);

public:  // State

    /**
     *  @return true when all queues in multi-queue are activated.
     */
    virtual bool is_active() const;

    /**
     *  @return true when queue is activated.
     */
    virtual bool is_active(int id) const;

    /**
     *  @return true if all queues are empty.
     */
    virtual bool is_empty() const;

    /**
     *  @return true if specified queue is empty.
     */
    virtual bool is_empty(int id) const;

    /**
     *  @return number of messages in all queues.
     */
    virtual size_t size() const;

    /**
     *  @return number of messages in the queue specified by id.
     */
    virtual size_t size(int id) const;

    /**
     *  @return number of registered queues.
     */
    virtual size_t queue_count() const;

#ifdef TPL_DEBUG

    /**
     *  @return number of messages in all queues (non-locking).
     */
    size_t size_debug() const;

    /**
     *  @return number of messages in the queue specified by id (non-locking).
     */
    size_t size_debug(int id) const;

    /**
     *  @return number of registered queues (non-locking).
     */
    size_t queues_count_debug() const;

#endif // TPL_DEBUG

private:

    using QueuePanel<T>::size_nl;

    size_t size_nl() const;

    void get_wait(unsigned timeout);

    T* round_robin_get(unsigned timeout);

    T* round_robin_try_get();

    std::list<T*> round_robin_get_bulk(size_t bulk_size, unsigned timeout);

    std::list<T*> round_robin_try_get_bulk(size_t bulk_size);

public:  // Printing

    friend std::ostream& operator<< <T>(std::ostream& os, const MultiQueue<T>& obj);

protected:

    virtual void print_on(std::ostream& os) const;

private:  // Attributes

    std::vector<int> _id_list;
    int _id_index;

    // mutex for create/remove sync. and get.
    mutable Mutex _mutex;
    Condition _get_cond;

    size_t _active_count; 	// number of active sub-queues
    size_t _get_waiting; 	// waiting queue size for get.
};

};  // namespace tpl

#ifndef NO_COMPILE_INSTANTIATE
#include "multi_queue.cc"
#endif

#endif // __TPL_MULTI_QUEUE_HH__
