// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#include "libtpl/thread/lock_guards.hh"
#include "libtpl/debug.hh"

using namespace tpl;

////////////////////////////////////////////////////////////
//                      MultiQueue<T>                     //
////////////////////////////////////////////////////////////

template <class T>
MultiQueue<T>::MultiQueue() : QueuePanel<T>(_mutex),
    _id_index(0), _get_cond(_mutex), _active_count(0), _get_waiting(0)
{
}

template <class T>
MultiQueue<T>::~MultiQueue()
{
}

template <class T>
bool
MultiQueue<T>::put(int id, T* element, unsigned timeout)
{
    TPL_ASSERT(element);

    LockGuard<Mutex> guard(_mutex);
    if (_active_count > 0  && put_nl(id, element, timeout)) {
    	_get_cond.signal();
    	return true;
    }
    return false;
}

template <class T>
bool
MultiQueue<T>::try_put(int id, T* element)
{
    TPL_ASSERT(element);

    LockGuard<Mutex> guard(_mutex);
    if (_active_count > 0 && try_put_nl(id, element)) {
    	_get_cond.signal();
    	return true;
    }
    return false;
}

template <class T>
bool
MultiQueue<T>::put_bulk(int id, std::list<T*> elements, unsigned timeout)
{
    TPL_ASSERT(!elements.empty());

    LockGuard<Mutex> guard(_mutex);
    if (_active_count > 0 && put_bulk_nl(id, elements, timeout)) {
    	_get_cond.signal();
    	return true;
    }
    return false;
}

template <class T>
bool
MultiQueue<T>::try_put_bulk(int id, std::list<T*> elements)
{
    TPL_ASSERT(!elements.empty());

    LockGuard<Mutex> guard(_mutex);
    if (_active_count > 0 && try_put_bulk_nl(id, elements)) {
    	_get_cond.signal();
    	return true;
    }
    return false;
}

template <class T>
T*
MultiQueue<T>::get(unsigned timeout)
{
    LockGuard<Mutex> guard(_mutex);
    if (_active_count > 0) {
    	return round_robin_get(timeout);
    }
    return 0;
}

template <class T>
T*
MultiQueue<T>::try_get()
{
    LockGuard<Mutex> guard(_mutex);
    if (_active_count > 0) {
    	return round_robin_try_get();
    }
    return 0;
}

template <class T>
std::list<T*>
MultiQueue<T>::get_bulk(size_t bulk_size, unsigned timeout)
{
    LockGuard<Mutex> guard(_mutex);
    if (_active_count > 0) {
    	return round_robin_get_bulk(bulk_size, timeout);
    }
    return std::list<T*>();
}

template <class T>
std::list<T*>
MultiQueue<T>::try_get_bulk(size_t bulk_size)
{
    LockGuard<Mutex> guard(_mutex);
    if (_active_count > 0) {
    	return round_robin_try_get_bulk(bulk_size);
    }
    return std::list<T*>();
}

template <class T>
int
MultiQueue<T>::create_queue(size_t max)
{
    LockGuard<Mutex> guard(_mutex);
    int id = this->create_queue_nl(max);
    _id_list.push_back(id);
    _active_count++;
    return id;
}

template <class T>
void
MultiQueue<T>::remove_queue(int id)
{
    TPL_ASSERT(id != 0);

    LockGuard<Mutex> guard(_mutex);
    if (this->is_active_nl(id))
	_active_count--;
    this->remove_queue_nl(id);

    std::vector<int>::iterator i = find(_id_list.begin(), _id_list.end(), id);
    if (i != _id_list.end())
    	_id_list.erase(i);
}

template <class T>
void
MultiQueue<T>::reset_all()
{
    LockGuard<Mutex> guard(_mutex);

    std::vector<int>::const_iterator beg = _id_list.begin();
    while (beg != _id_list.end()) {
        this->reset_nl(*beg);
        ++beg;
    }
}

template <class T>
void
MultiQueue<T>::reset(int id)
{
    TPL_ASSERT(id != 0);

    LockGuard<Mutex> guard(_mutex);
    this->reset_nl(id);
}

template <class T>
void
MultiQueue<T>::activate_all()
{
    LockGuard<Mutex> guard(_mutex);

    std::vector<int>::const_iterator beg = _id_list.begin();
    while (beg != _id_list.end()) {
        this->activate_nl(*beg);
        ++beg;
    }
    _active_count = _id_list.size();
}

template <class T>
void
MultiQueue<T>::activate(int id)
{
    TPL_ASSERT(id != 0);

    LockGuard<Mutex> guard(_mutex);
    if (!this->is_active_nl(id))
    	_active_count++;
    this->activate_nl(id);
}

template <class T>
void
MultiQueue<T>::deactivate_all()
{
    LockGuard<Mutex> guard(_mutex);
    _active_count = 0;

    std::vector<int>::const_iterator beg = _id_list.begin();
    while (beg != _id_list.end()) {
    	this->deactivate_nl(*beg);
    	++beg;
    }

    while(_get_waiting) {
    	_get_cond.broadcast();
    	_get_cond.wait(100);
    }
}

template <class T>
void
MultiQueue<T>::deactivate(int id)
{
    TPL_ASSERT(id != 0);

    LockGuard<Mutex> guard(_mutex);
    if (this->is_active_nl(id))
    	_active_count--;
    this->deactivate_nl(id);
    if (_get_waiting)
    	_get_cond.broadcast();
}

template <class T>
void
MultiQueue<T>::clear_all()
{
    LockGuard<Mutex> guard(_mutex);

    std::vector<int>::const_iterator beg = _id_list.begin();
    while (beg != _id_list.end()) {
    	this->clear_nl(*beg);
    	++beg;
    }
}

template <class T>
void
MultiQueue<T>::clear(int id)
{
    TPL_ASSERT(id != 0);

    LockGuard<Mutex> guard(_mutex);
    this->clear_nl(id);
}

template <class T>
size_t
MultiQueue<T>::max_elements(int id) const
{
    TPL_ASSERT(id != 0);

    LockGuard<Mutex> guard(_mutex);
    return this->max_elements_nl(id);
}

template <class T>
void
MultiQueue<T>::set_max_elements(int id, size_t max)
{
    TPL_ASSERT(id != 0);

    LockGuard<Mutex> guard(_mutex);
    this->set_max_elements_nl(id, max);
}

template <class T>
bool
MultiQueue<T>::is_active() const
{
    return _active_count > 0;
}

template <class T>
bool
MultiQueue<T>::is_active(int id) const
{
    LockGuard<Mutex> guard(_mutex);
    return this->is_active_nl(id);
}

template <class T>
bool
MultiQueue<T>::is_empty() const
{
    LockGuard<Mutex> guard(_mutex);
    std::vector<int>::const_iterator beg = _id_list.begin();
    while (beg != _id_list.end()) {
        if (!this->is_empty_nl(*beg)) return false;
        ++beg;
    }
    return true;
}

template <class T>
bool
MultiQueue<T>::is_empty(int id) const
{
    LockGuard<Mutex> guard(_mutex);
    return this->is_empty_nl(id);
}

template <class T>
size_t
MultiQueue<T>::size() const
{
    LockGuard<Mutex> guard(_mutex);
    return this->size_nl();
}

template <class T>
size_t
MultiQueue<T>::size(int id) const
{
    LockGuard<Mutex> guard(_mutex);
    return this->size_nl(id);
}

template <class T>
size_t
MultiQueue<T>::queue_count() const
{
    LockGuard<Mutex> guard(_mutex);
    return this->queue_count_nl();
}

template <class T>
size_t
MultiQueue<T>::size_nl() const
{
    size_t sz = 0;
    std::vector<int>::const_iterator beg = _id_list.begin();
    while (beg != _id_list.end()) {
        sz += this->size_nl(*beg);
        ++beg;
    }
    return sz;
}

#ifdef TPL_DEBUG

template <class T>
int
MultiQueue<T>::size_debug() const
{
    return this->size_nl();
}

template <class T>
int
MultiQueue<T>::size_debug(int id) const
{
    return this->size_nl(id);
}

template <class T>
int
MultiQueue<T>::queues_count_debug() const
{
    return this->queues_count_nl();
}

#endif // TPL_DEBUG

template <class T>
void
MultiQueue<T>::get_wait(unsigned timeout)
{
    _get_waiting++;
    _get_cond.wait(timeout);
    _get_waiting--;
}

template <class T>
void
MultiQueue<T>::print_on(std::ostream&) const
{}

template <class T>
T*
MultiQueue<T>::round_robin_get(unsigned timeout)
{
    int i;

    if (_id_list.size() == 0) {
    	get_wait(timeout);
    	timeout = 0;
    }

    bool is_empty = true;
    bool is_expired = false;
    while (is_empty) {
    	int sz = _id_list.size();
    	if (_active_count == 0 || sz == 0) return 0;
    	i = _id_index = (_id_index < sz)? _id_index : 0;

    	do {
    	    int id = _id_list[i];
    	    TPL_ASSERT(id != 0);
    	    if (this->is_active_nl(id) && !this->is_empty_nl(id)) {
    	    	is_empty = false;
    	    	break;
    	    }

    	    if (++i >= sz)
    	    	i = 0;
    	} while (i != _id_index);

    	if (is_empty) {
    	    if (is_expired) return 0;
    	    get_wait(timeout);
    	    is_expired = true; // protect ourself from infinite loop in get()
    	}
    }

    _id_index = i + 1;
    return this->try_get_nl(_id_list[i]);
}

template <class T>
T*
MultiQueue<T>::round_robin_try_get()
{
    if (_id_list.size() == 0)
    	return 0;

    bool is_empty = true;
    int sz = _id_list.size();
    int i = _id_index = (_id_index < sz)? _id_index : 0;

    do {
    	int id = _id_list[i];
    	TPL_ASSERT(id != 0);
    	if (this->is_active_nl(id) && !this->is_empty_nl(id)) {
    	    is_empty = false;
    	    break;
    	}

    	if (++i >= sz)
    	    i = 0;
    } while (i != _id_index);

    if (is_empty) return 0;

    _id_index = i + 1;
    return this->try_get_nl(_id_list[i]);
}

template <class T>
std::list<T*>
MultiQueue<T>::round_robin_get_bulk(size_t bulk_size, unsigned timeout)
{
    UNUSED(bulk_size);
    int i;
    std::list<T*> lst;

    if (_id_list.size() == 0) {
    	get_wait(timeout);
    	timeout = 0;
    if (_id_list.size() == 0)
    	return lst;
    }

    int sz = 0;
    bool is_empty = true;
    while (is_empty) {
    	sz = _id_list.size();
    	i = _id_index = (_id_index < sz)? _id_index : 0;

    	do {
    	    int id = _id_list[i];
    	    if (this->is_active_nl(id) && !this->is_empty_nl(id)) {
    	    	is_empty = false;
    	    	break;
    	    }

    	    if (++i >= sz)
    	    	i = 0;
    	} while (i != _id_index);

    	if (is_empty) {
    	    if (timeout == 0) return lst;
    	    get_wait(timeout);
    	    timeout = 0; // protect ourself from infinite loop in get()
    	}
    }

    int empty_count = 0;
    do {
    	if (i >= sz) {
    	    empty_count = 0;
    	    i = 0;
    	}

    	int id = _id_list[i];
    	if (this->is_active_nl(id) && !this->is_empty_nl(id)) {
    	    _id_index = i;
    	    lst.push_back(this->try_get_nl(id));
    	}
    	else {
    	    ++empty_count;
    	}
    	++i;
    } while (empty_count < sz);

    ++_id_index;
    return lst;
}

template <class T>
std::list<T*>
MultiQueue<T>::round_robin_try_get_bulk(size_t bulk_size)
{
    UNUSED(bulk_size);
    std::list<T*> lst;

    if (_id_list.size() == 0)
    	return lst;

    bool is_empty = true;
    int sz = _id_list.size();
    int i = _id_index = (_id_index < sz)? _id_index : 0;

    do {
    	int id = _id_list[i];
    	if (this->is_active_nl(id) && !this->is_empty_nl(id)) {
    	    is_empty = false;
    	    break;
    	}

    	if (++i >= sz)
    	    i = 0;
    } while (i != _id_index);

    if (is_empty)
    	return lst;

    int empty_count = 0;
    do {
    	if (i >= sz) {
    	    empty_count = 0;
    	    i = 0;
    	}

    	int id = _id_list[i];
    	if (this->is_active_nl(id) && !this->is_empty_nl(id)) {
    	    _id_index = i;
    	    lst.push_back(this->try_get_nl(id));
    	}
    	else {
    	    ++empty_count;
    	}
    	++i;
    } while (empty_count < sz);

    ++_id_index;
    return lst;
}

