// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#include "libtpl/thread/lock_guards.hh"
#include "libtpl/debug.hh"

using namespace tpl;

////////////////////////////////////////////////////////////
//                LinkedBlockingQueue<T>                  //
////////////////////////////////////////////////////////////

template <class T>
LinkedBlockingQueue<T>::LinkedBlockingQueue(size_t max) :
    _max_elements(max), _mutex(new Mutex), _get_cond(*_mutex), _put_cond(*_mutex),
    _shutdown(false), _shared_mutex(false), _put_waiting(0), _get_waiting(0)
{
}

template <class T>
LinkedBlockingQueue<T>::LinkedBlockingQueue(Mutex* mutex, size_t max) :
    _max_elements(max), _mutex(mutex), _get_cond(*_mutex), _put_cond(*_mutex),
    _shutdown(false), _shared_mutex(true), _put_waiting(0), _get_waiting(0)
{
}

template <class T>
LinkedBlockingQueue<T>::~LinkedBlockingQueue()
{
    clear_nl();
    if (!_shared_mutex) delete _mutex;
}

template <class T>
bool
LinkedBlockingQueue<T>::put(T* element, unsigned timeout)
{
    TPL_ASSERT(element);
    LockGuard<Mutex> guard(*_mutex);
    return put_nl(element, timeout);
}

template <class T>
bool
LinkedBlockingQueue<T>::try_put(T* element)
{
    TPL_ASSERT(element);
    LockGuard<Mutex> guard(*_mutex);
    return try_put_nl(element);
}

template <class T>
bool
LinkedBlockingQueue<T>::put_bulk(std::list<T*> elements, unsigned timeout)
{
    TPL_ASSERT(!elements.empty());
    LockGuard<Mutex> guard(*_mutex);
    return put_bulk_nl(elements, timeout);
}

template <class T>
bool
LinkedBlockingQueue<T>::try_put_bulk(std::list<T*> elements)
{
    TPL_ASSERT(!elements.empty());
    LockGuard<Mutex> guard(*_mutex);
    return try_put_bulk_nl(elements);
}

template <class T>
T*
LinkedBlockingQueue<T>::get(unsigned timeout)
{
    LockGuard<Mutex> guard(*_mutex);
    return get_nl(timeout);
}

template <class T>
T*
LinkedBlockingQueue<T>::try_get()
{
    LockGuard<Mutex> guard(*_mutex);
    return try_get_nl();
}

template <class T>
std::list<T*>
LinkedBlockingQueue<T>::get_bulk(size_t bulk_size, unsigned timeout)
{
    LockGuard<Mutex> guard(*_mutex);
    return get_bulk_nl(bulk_size, timeout);
}

template <class T>
std::list<T*>
LinkedBlockingQueue<T>::try_get_bulk(size_t bulk_size)
{
    LockGuard<Mutex> guard(*_mutex);
    return try_get_bulk_nl(bulk_size);
}

template <class T>
void
LinkedBlockingQueue<T>::clear()
{
    LockGuard<Mutex> guard(*_mutex);
    clear_nl();
}

template <class T>
void
LinkedBlockingQueue<T>::activate()
{
    _shutdown = false;
}

template <class T>
void
LinkedBlockingQueue<T>::deactivate()
{
    LockGuard<Mutex> guard(*_mutex);
    deactivate_nl();
}

template <class T>
void
LinkedBlockingQueue<T>::reset()
{
    LockGuard<Mutex> guard(*_mutex);
    reset_nl();
}

template <class T>
void
LinkedBlockingQueue<T>::set_max_elements(size_t max)
{
    _max_elements = max;
}

template <class T>
size_t
LinkedBlockingQueue<T>::max_elements() const
{
    return _max_elements;
}

template <class T>
bool
LinkedBlockingQueue<T>::is_active() const
{
    return !_shutdown;
}

template <class T>
bool
LinkedBlockingQueue<T>::is_empty() const
{
    LockGuard<Mutex> guard(*_mutex);
    return is_empty_nl();
}

template <class T>
size_t
LinkedBlockingQueue<T>::size() const
{
    LockGuard<Mutex> guard(*_mutex);
    return size_nl();
}

#ifdef TPL_DEBUG

template <class T>
bool
LinkedBlockingQueue<T>::is_locked() const
{
    bool locked = _mutex->trylock();
    if (locked)
        _mutex->unlock();

    return !locked;
}

template <class T>
size_t
LinkedBlockingQueue<T>::size_debug() const
{
    return _queue.size();
}

template <class T>
pthread_t
LinkedBlockingQueue<T>::owner_thread() const
{
    return _mutex->owner_thread();
}

#endif // TPL_DEBUG

template <class T>
bool
LinkedBlockingQueue<T>::put_nl(T* element, unsigned timeout)
{
    TPL_ASSERT(element);

    if (!_shutdown) {
    	if (_max_elements > 0) {
    	    while (_queue.size() >= _max_elements) {
    	    	if (timeout == 0) {
    	    	        ++_put_waiting;
    	    	        _put_cond.wait();
    	    	        --_put_waiting;
    	    	        if (_shutdown) return false;
    	    	}
    	    	else {
    	    	    ++_put_waiting;
    	    	    _put_cond.wait(timeout);
    	    	    --_put_waiting;
    	    	    if (_queue.size() >= _max_elements || _shutdown) {
    	    	    	return false;
    	    	    }
    	    	}
    	    }
    	}

    	_queue.push_back(element);
    	_get_cond.signal();
    	return true;
    }

    return false;
}

template <class T>
bool
LinkedBlockingQueue<T>::try_put_nl(T* element)
{
    TPL_ASSERT(element);

    if (!_shutdown) {
    	if (_max_elements > 0 && _queue.size() >= _max_elements)
    	    return false;

    	_queue.push_back(element);
    	_get_cond.signal();
    	return true;
    }

    return false;
}

template <class T>
bool
LinkedBlockingQueue<T>::put_bulk_nl(std::list<T*> elements, unsigned timeout)
{
    TPL_ASSERT(!elements.empty());

    if (!_shutdown) {
    	if (_max_elements > 0) {
    	    while ((_queue.size() +  elements.size()) > _max_elements) {
    	    	if (timeout == 0) {
    	    	    ++_put_waiting;
    	    	    _put_cond.wait();
    	    	    --_put_waiting;
    	    	    if (_shutdown) return false;
    	    	}
    	    	else {
    	    	    ++_put_waiting;
    	    	    _put_cond.wait(timeout);
    	    	    --_put_waiting;
    	    	    if (_queue.size() >= _max_elements || _shutdown) {
    	    	    	return false;
    	    	    }
    	    	}
    	    }
    	}

    	_queue.splice(_queue.end(), elements);
    	_get_cond.signal();
    	return true;
    }

    return false;
}

template <class T>
bool
LinkedBlockingQueue<T>::try_put_bulk_nl(std::list<T*> elements)
{
    TPL_ASSERT(!elements.empty());

    if (!_shutdown) {
    	if (_max_elements > 0 && (_queue.size() + elements.size()) > _max_elements)
    	    return false;

    	_queue.splice(_queue.end(), elements);
    	_get_cond.signal();
    	return true;
    }

    return false;
}

template <class T>
T*
LinkedBlockingQueue<T>::get_nl(unsigned timeout)
{
    if (!_shutdown) {
    	while (_queue.empty()) {
    	    if (timeout == 0) {
    	    	++_get_waiting;
    	    	_get_cond.wait();
    	    	--_get_waiting;
    	    	if (_shutdown) return 0;
    	    }
    	    else {
    	    	++_get_waiting;
    	    	_get_cond.wait(timeout);
    	    	--_get_waiting;
    	    	if (_queue.size() == 0 || _shutdown) {
    	    	    return 0;
    	    	}
    	    }
    	}

    	T* element = _queue.front();
    	_queue.pop_front();

    	if (_max_elements > 0)
    	    _put_cond.signal();

    	return element;
    }

    return 0;
}

template <class T>
T*
LinkedBlockingQueue<T>::try_get_nl()
{
    if (!_shutdown) {
    	if (_queue.empty()) return 0;

    	T* element = _queue.front();
    	_queue.pop_front();

    	if (_max_elements > 0)
    	    _put_cond.signal();

    	return element;
    }

    return 0;
}

template <class T>
std::list<T*>
LinkedBlockingQueue<T>::get_bulk_nl(size_t bulk_size, unsigned timeout)
{
    std::list<T*> lst;

    if (!_shutdown) {
    	while (_queue.size() < bulk_size) {
    	    if (timeout == 0) {
    	    	++_get_waiting;
    	    	_get_cond.wait();
    	    	--_get_waiting;
    	    	if (_shutdown) return lst;
    	    }
    	    else {
    	    	++_get_waiting;
    	    	_get_cond.wait(timeout);
    	    	--_get_waiting;
    	    	if (_queue.size() == 0 || _shutdown) {
    	    	    return lst;
    	    	}
    	    }
    	}

    	typename std::list<T*>::iterator pos = _queue.begin();
    	for (size_t i = 0; i <= bulk_size; ++i) ++pos;

    	lst.splice(lst.begin(), _queue, _queue.begin(), pos);

    	if (_max_elements > 0)
    	    _put_cond.signal();
    }

    return lst;
}

template <class T>
std::list<T*>
LinkedBlockingQueue<T>::try_get_bulk_nl(size_t bulk_size)
{
    std::list<T*> lst;

    if (!_shutdown) {
    	if (_queue.size() < bulk_size) return lst;

    	typename std::list<T*>::iterator pos = _queue.begin();
    	for (size_t i = 0; i <= bulk_size; ++i) ++pos;

    	lst.splice(lst.begin(), _queue, _queue.begin(), pos);

    	if (_max_elements > 0)
    	    _put_cond.signal();
    }

    return lst;
}

template <class T>
void
LinkedBlockingQueue<T>::clear_nl()
{
    typename std::list<T*>::iterator i = _queue.begin();
    typename std::list<T*>::iterator end = _queue.end();
    while (i != end) {
        delete *i;
        i++;
    }
    _queue.clear();
}

template <class T>
void
LinkedBlockingQueue<T>::reset_nl()
{
    _put_cond.broadcast();
    _get_cond.broadcast();
}

template <class T>
void
LinkedBlockingQueue<T>::deactivate_nl()
{
    _shutdown = true;

    while(_put_waiting) {
    	_put_cond.broadcast();
    	_put_cond.wait(100);
    }

    while(_get_waiting) {
    	_get_cond.broadcast();
    	_get_cond.wait(100);
    }
}

template <class T>
bool
LinkedBlockingQueue<T>::is_empty_nl() const
{
    return _queue.empty();
}

template <class T>
size_t
LinkedBlockingQueue<T>::size_nl() const
{
    return _queue.size();
}

template <class T>
void
LinkedBlockingQueue<T>::print_on(std::ostream& os) const
{
    UNUSED(os);
}
