// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_BLOCKING_QUEUE_HH__
#define __TPL_BLOCKING_QUEUE_HH__

#include "libtpl/tpl_decls.hh"
#include <iostream>
#include <list>

namespace tpl {

template <class T>
class BlockingQueue;

template <class T>
inline std::ostream& operator<< (std::ostream& os, const BlockingQueue<T>& obj)
{
    obj.print_on(os);
    return os;
}

/**
 *  Blocking queue interface.
 */
template <class T>
class BlockingQueue
{
public:  // Create/Copy/Destroy

    BlockingQueue();
    virtual ~BlockingQueue() = 0;

    TPL_NO_COPY_TEMPLATE(BlockingQueue, T);

public:  // Usage

    /**
     *  Put element into the queue.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is saturated the call will block
     *  and contend for the first available slot in the queue after
     *  get() operation is called.
     *  @return true if element has been inserted into the queue.
     */
    virtual bool put(T* element, unsigned timeout = 0) = 0;

    /**
     *  Try to put element into the queue without blocking.
     *  XXX: The call will return immediately if the queue is saturated.
     *  @return true if element has been inserted into the queue.
     */
    virtual bool try_put(T* element) = 0;

    /**
     *  Put multiple elements into the queue in bulk mode.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is saturated the call will block
     *  and contend for the elements.size() available slots in the queue after
     *  get() operation is called.
     *  @return true if all elements have been inserted into the queue.
     */
    virtual bool put_bulk(std::list<T*> elements, unsigned timeout = 0) = 0;

    /**
     *  Try to put multiple elements into the queue in bulk mode.
     *  XXX: The call will return immediately if the queue is saturated.
     *  @return true if all elements have been inserted into the queue.
     */
    virtual bool try_put_bulk(std::list<T*> elements) = 0;

    /**
     *  Get element from the queue.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is empty the call will block
     *  and contend for the first available element in the queue after
     *  put() operation is called.
     *  @return pointer to the element withdrawn from the queue
     *  or NULL if the queue was empty.
     */
    virtual T* get(unsigned timeout = 0) = 0;

    /**
     *  Try to get an element from the queue.
     *  XXX: The call will return immediately if the queue is empty.
     *  @return pointer to the element withdrawn from the queue
     *  or NULL if the queue was empty.
     */
    virtual T* try_get() = 0;

    /**
     *  Get multiple elements from the queue in bulk mode.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout == 0 and the queue is empty the call will block
     *  and contend for the first bulk_size available elements in the queue after
     *  put() operation is called.
     *  @return a list of pointers to the elements withdrawn from the queue
     *  or an empty list if the queue was empty.
     */
    virtual std::list<T*> get_bulk(size_t bulk_size, unsigned timeout = 0) = 0;

    /**
     *  Try to get multiple elements from the queue in bulk mode without blocking.
     *  XXX: The call will return immediately if the queue is empty.
     *  @return a list of pointers to the elements withdrawn from the queue
     *  or an empty list if the queue was empty.
     */
    virtual std::list<T*> try_get_bulk(size_t bulk_size) = 0;

public:  // Management

    /**
     *  Clears the queue.
     */
    virtual void clear() = 0;

    /**
     *  Activates the queue.
     */
    virtual void activate() = 0;

    /**
     *  Deactivates the queue.
     */
    virtual void deactivate() = 0;

public:  // State

    /**
     *  Returns true when queue is not in shutdown state.
     */
    virtual bool is_active() const = 0;

    /**
     *  Returns true when queue is empty.
     */
    virtual bool is_empty() const = 0;

    /**
     *  Returns current number of elements in the queue.
     */
    virtual size_t size() const = 0;

public:  // Printing

    friend std::ostream& operator<< <T>(std::ostream& os, const BlockingQueue<T>& obj);

protected:

    virtual void print_on(std::ostream& os) const = 0;

};

// default constructor implementation
template <class T>
BlockingQueue<T>::BlockingQueue()
{
}

// pure virtual destructor implementation
template <class T>
BlockingQueue<T>::~BlockingQueue()
{
}

};  // namespace tpl

#endif // __TPL_BLOCKING_QUEUE_HH__
