// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_TASK_MANAGER_HH__
#define __TPL_TASK_MANAGER_HH__

#include "libtpl/thread/mutex.hh"
#include "libtpl/thread/condition.hh"
#include "libtpl/thread/monitor.hh"
#include "libtpl/thread/thread.hh"

#include <map>
#include <list>

namespace tpl {

class TaskBase;

/**
 *  Task manager entity
 */
class TaskManager : public Monitor
{
    typedef std::map<int, TaskBase*, std::less<int> > TaskMap;
    typedef std::list<TaskBase*> TempList;

public:     //// Create/Copy/Destroy

    /**
     *  Create TaskManager instance (singleton)
     */
    static TaskManager* instance();

    virtual ~TaskManager();

protected:
     //  Default constructor.
     //  XXX: Protected to ensure Singleton pattern usage.
    TaskManager();

    TPL_NO_COPY(TaskManager);

public:  // Usage

    /**
     *  Add new task
     */
    void add(TaskBase* task);

    /**
     *  Remove task
     */
    void remove(TaskBase* task);

    /**
     *  Join all active threads
     */
    void join_all();

protected:  // Runtime

    /**
     *  Internal cycle start
     */
    void start();

    /**
     *  Garbage collector cycle
     */
    void run();

public:  // Printing

    friend inline std::ostream& operator<< (std::ostream& os, const TaskManager& obj);

protected:

    virtual void print_on(std::ostream& os) const;

private:

    /**  Thread function */
    static void* thread_proc(void* arg);

    static Mutex _inst_mutex;		// Instance mutex, guards instantiation
    static TaskManager* _instance;		// TaskPool instance pointer

    bool _shutdown;
    TaskMap _task_map;			// Active tasks collection
    TempList _temp_list;		// Temporary tasks collection
    mutable Condition _cond;	// Condition for cleanup cycle
    Thread _thread;
};

inline std::ostream& operator<< (std::ostream& os, const TaskManager& obj)
{
    obj.print_on ( os );
    return os;
}

};  // namespace tpl

#endif // __TPL_TASK_MANAGER_HH__
