// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#include "task_mgr.hh"
#include "task.hh"
#include "libtpl/thread/lock_guards.hh"
#include "libtpl/debug.hh"

using namespace tpl;

////////////////////////////////////////////////////////////
//                    TaskManager                         //
////////////////////////////////////////////////////////////

Mutex TaskManager::_inst_mutex;
TaskManager* TaskManager::_instance = 0;

TaskManager*
TaskManager::instance()
{
    LockGuard<Mutex> guard(_inst_mutex);
    if (_instance == 0) {
    	_instance = new TaskManager();
    	_instance->start();
    }
    return _instance;
}

TaskManager::TaskManager() : _shutdown(false), _cond(mutex())
{
}

TaskManager::~TaskManager()
{
}

void
TaskManager::add(TaskBase* task)
{
    _task_map[task->id()] = task;
}

void
TaskManager::remove(TaskBase* task)
{
    int id = task->id();
    if (_task_map.count(id) == 1) {
        TaskBase* task = _task_map[id];
        if (task->mode() == TASK_TEMPORARY) {
            _temp_list.push_back(task);
            _cond.signal();
        }
        _task_map.erase(id);
    }
}

void
TaskManager::join_all()
{
    void *retval;

    do {
    	LockGuard<TaskManager> guard(*this);
    	TaskMap::iterator i = _task_map.begin();
    	TaskMap::iterator end = _task_map.end();
    	while (i != end) {
    	    TaskBase* task = i->second;
    	    if (task->mode() != TASK_TEMPORARY) {
    	    	Thread& th = task->thread();
    	    	while (true) {
    	    	    int err = th.join(&retval);
    	    	    if (err == ERROR_INTERRUPTED) {
    	    	    	continue;
    	    	    }
    	    	    TPL_ASSERT(err == 0);
    	    	    break;
    	    	}
    	    	_task_map.erase(i++);
    	    } else
    	    	i++;
    	}
    } while(0);

    // notify garbage collector thread that it needs to shutdown
    // and wait while it exits
    do {
    	LockGuard<TaskManager> guard(*this);
    	_shutdown = true;
    	_cond.signal();
    } while(0);

    while (true) {
    	int err = _thread.join(&retval);
    	if (err == ERROR_INTERRUPTED)
    	    continue;
    	TPL_ASSERT(err == 0);
    	break;
    }
}

void
TaskManager::start()
{
    LockGuard<TaskManager> guard(*this);

    _thread.set_thread_function(TaskManager::thread_proc);
    _thread.start(this);

    // ensure garbage collector thread started
    while (!_shutdown)
        if (_cond.wait() != ERROR_INTERRUPTED)
            break;
}

void
TaskManager::run()
{
    // notify parent thread we started
    lock();
    _cond.broadcast();
    while (!(_shutdown && _task_map.empty() && _temp_list.empty())) {
    	if (_temp_list.empty()) {
    	    _cond.wait();
    	}

    	TempList::iterator beg = _temp_list.begin();
    	TempList::iterator end = _temp_list.end();
    	// unlocking mutex to allow threads to exit
    	unlock();

    	while (beg != end) {
    	    TaskBase* t = *(beg++);
    	    t->join();
    	    delete t;
    	}

        // lock mutex again to proceed with list changes
        lock();
    	_temp_list.erase(_temp_list.begin(), end);
    }
    // notify waiting thread we stopped
    unlock();
}

void*
TaskManager::thread_proc(void *ctx)
{
    TaskManager* p = static_cast<TaskManager*>(ctx);
    try {
        p->run();
    } catch (...) {
    }

    return 0;
}

void
TaskManager::print_on(std::ostream&) const
{
}

