// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_TASK_HH__
#define __TPL_TASK_HH__

#include "libtpl/thread/thread.hh"

#include <iostream>

namespace tpl {

enum TaskMode {
    TASK_REGULAR,
    TASK_DETACHED,
    TASK_TEMPORARY
};

class TaskManager;

/**
 *  @short Task objects interface
 */
class TaskBase
{
public:  // Types

    typedef Thread::thread_id_t thread_id_t;
    friend class TaskManager;

public:  // Create/Copy/Destroy

    TaskBase() {}

    virtual ~TaskBase() = 0;

    TPL_NO_COPY(TaskBase);

public:  // Usage

    /**
     *  Starts task thread
     */
    virtual void start() = 0;

    /**
     *  Joins task thread
     */
    virtual void join() = 0;

protected:

    /**
     *  @return task thread reference.
     */
    virtual Thread& thread() = 0;

public:  // Properties

    /**
     *  Returns thread id
     */
    virtual thread_id_t id() const = 0;

    /**
     *  Returnd task run mode: REGULAR, DETACHED or TEMPORARY
     */
    virtual TaskMode mode() const = 0;

public:  // Printing

    friend inline std::ostream& operator<< (std::ostream& os, const TaskBase& obj);

protected:

    virtual void print_on(std::ostream&) const = 0;

};

inline std::ostream& operator<< (std::ostream& os, const TaskBase& obj)
{
    obj.print_on ( os );
    return os;
}


/**
 *
 */
template <class LIFE_CYCLE>
class Task : public TaskBase, public LIFE_CYCLE
{
public:     //// Create/Copy/Destroy

    /**
     *  Create thread object.
     *  XXX: Thread object is initialized, however thread is not started.
     */
    Task(TaskMode mode = TASK_REGULAR);

    /**
     *  Destroy thread object.
     *  XXX: Does implicit shutdown for THREAD_STAND_BY policy.
     */
    virtual ~Task();

    TPL_NO_COPY_TEMPLATE(Task, LIFE_CYCLE);

public:  // Usage

    /**
     *  Starts task thread.
     */
    virtual void start();

    /**
     *  Join task thread.
     */
    virtual void join();

protected:

    /**
     *  @return task thread reference.
     */
    virtual Thread& thread() {
    	return _thread;
    }

public:  // State

    /**
     *  @return thread id of task thread
     */
    virtual thread_id_t id() const {
    	return _thread.id();
    }

    /**
     *  @return task run mode.
     */
    virtual TaskMode mode() const {
    	return _mode;
    }

    using LIFE_CYCLE::is_running;
    using LIFE_CYCLE::is_shutdown;

protected:

    /**
     *  Task thread function.
     *  Notes: This method must be implemented by concrete Task. As soon as Task
     *  is started run() method is called in a dedicated thread.
     */
    virtual void run() = 0;

    virtual void cleanup();

protected:  // Printing

    virtual void print_on(std::ostream& os) const;

private:

    TaskMode _mode;
    Thread _thread;
};

};  // namespace tpl

#endif //__TPL_TASK_HH__
