// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_SERVICE_HH__
#define __TPL_SERVICE_HH__

#include "task.hh"
#include "task_policy.hh"
#include "libtpl/queues/blocking_queue.hh"

#include <iostream>

namespace tpl {

/**
 *  Service = Task + BlockingQueue
 *  XXX: Implements worker pattern.
 */
template <class T, class LIFE_CYCLE = RUN_THREAD_ONCE>
class Service : public Task<LIFE_CYCLE>
{
public:  // Create/Copy/Destroy

    Service(size_t queue_limit = 0);

    Service(BlockingQueue<T>* queue);

    virtual ~Service();

    TPL_NO_COPY_TEMPLATE(Service, T, LIFE_CYCLE);

public:  // Queue interface

    /**
     *  Put message into the queue. 
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout <= 0 it becomes blocking call.
     */
    virtual bool put(T* msg, int timeout = -1);

    /**
     *  Put messages list into the queue in bulk mode.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout <= 0 it becomes blocking call.
     */
    virtual bool put_bulk(std::list<T*> msg_bulk, int timeout = -1);

    /**
     *  Get message from the queue. 
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout <= 0 it becomes blocking call. 
     */
    virtual T* get(int timeout = -1);

    /**
     *  Get messages list from the queue in bulk mode.
     *  XXX: Call is blocking for the duration of the timeout.
     *  If timeout <= 0 it becomes blocking call. 
     */
    virtual std::list<T*> get_bulk(int timeout = -1);

public:  // Properties

    /**  Resets the working queue to an internal one. */
    void reset_queue(size_t queue_limit = 0);

    /**  Sets external working queue. */
    void set_queue(BlockingQueue<T>* queue);

    /**  @return pointer to the working queue. */
    BlockingQueue<T>* queue();

public:  // State

    /**
     *  Returns true when queue is empty.
     */
    bool is_empty() const;

    /**
     *  Returns number of messages in the queue 
     */
    size_t queue_size() const;

#ifdef TPL_DEBUG

    /**
     *  Returns true when queue is locked.
     */
    bool is_locked() const;

    /**
     *  Returns number of messages in the queue without locking
     */
    int queue_size_debug() const;

    /**
     *  Returns thread id of the host thread
     */
    int owner_thread() const;

#endif

protected:

    void activate_queue();

    void deactivate_queue();

protected:  // Printing

    virtual void print_on(std::ostream& os) const;

private:

    bool _shared_queue;
    BlockingQueue<T>* _queue;
};

};  // namespace tpl

#ifndef NO_COMPILE_INSTANTIATE
#include "service.cc"
#endif
#endif // __TPL_SERVICE_HH__
