// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_EXCEPTIONS_HH__
#define __TPL_EXCEPTIONS_HH__

#include <stdexcept>
#include <iostream>
#include <string>

#include "libtpl/utils/c_type.hh"

namespace tpl {

/**
 *  Base class for all TPL exceptions.
 *  XXX: This class should not be used to derive specific exceptions.
 *  Use ErrorFatal, ErrorRecoverable, ErrorClient instead.
 */
class ErrorBase : public std::exception
{
public:

    static const int ERROR_CODE_GENERIC = -1;
    enum ERROR_TYPE { ERROR_FATAL, ERROR_RECOVERABLE };

public:  // Create/Copy/Destroy

    ErrorBase(int type, const std::string& message, const char* file, size_t line, const char* function) :
    	_type(type), _code(ERROR_CODE_GENERIC), _message(message), _file(file), _line(line), _function(function) {}
    ErrorBase(int type, int code, const std::string& message, const char* file, size_t line, const char* function) :
    	_type(type), _code(code), _message(message), _file(file), _line(line), _function(function) {}
    ErrorBase(int type, int code, const char* file, size_t line, const char* function) :
    	_type(type), _code(code), _file(file), _line(line), _function(function) {
    	char* error;
    	error = ::strerror(code);
    	_message = std::string(error);
    }

    virtual ~ErrorBase() throw() {}

public:  // Usage

    /**
     *  Returnds error type.
     */
    int type() const { return _type; }

    /**
     *  Returns error code.
     */
    int code() const { return _code; }

    /**
     *	Description: Returnds error type.
     */
    virtual const char* what() const throw() { return _message.c_str(); }

    /**
     *  Returns the location where exception originates from.
     */
    const std::string where() const {
    	c_type<int> line(_line);
    	return (std::string(_file) + ":" + std::string(line) + " " + std::string(_function));
    }

    /**
     *  Returns error message.
     */
    virtual std::string as_string() const = 0;

public:     // Printing

    friend std::ostream& operator<< (std::ostream& os, const ErrorBase& obj)
    {
    	obj.print_on(os);
    	return os;
    }

protected:

    virtual void print_on(std::ostream& os) const
    {
    	os << "Exception: " << where() << ": " << what() << " [code: " << _code << "]";
    }

protected:

    int         _type;       // Exception type
    int         _code;       // Error code
    std::string _message;    // Error message.

    const char* _file;       // The file name where exception occured
    size_t      _line;       // The line number where exception occured
    const char* _function;   // Name of the function that produced an error.
};

/**
 *  The ERROR_FATAL exception class.
 */
class ErrorFatal : public ErrorBase
{
public:  // Create/Copy/Destroy

    ErrorFatal(const std::string& message, const char* file = "", size_t line = 0, const char* function = "") :
    	ErrorBase(ErrorBase::ERROR_FATAL, message, file, line, function) {}
    ErrorFatal(int code, const std::string& message, const char* file = "", size_t line = 0, const char* function = "") :
    	ErrorBase(ErrorBase::ERROR_FATAL, code, message, file, line, function) {}
    ErrorFatal(int code, const char* file = "", size_t line = 0, const char* function = "") :
    	ErrorBase(ErrorBase::ERROR_FATAL, code, file, line, function) {}

    virtual ~ErrorFatal() throw() {}

public:  // Usage

    virtual std::string as_string() const
    {
    	c_type<int> code(_code);
    	return ("FATAL ERROR: " + where() + ": " + std::string(what()) + " [code: " + std::string(code) +  "]");
    }
};

/**
 *  The ERROR_RECOVERABLE exception class.
 */
class ErrorRecoverable : public ErrorBase
{
public:  // Create/Copy/Destroy

    ErrorRecoverable(const std::string& message, const char* file = "", size_t line = 0, const char* function = "") :
    	ErrorBase(ErrorBase::ERROR_RECOVERABLE, message, file, line, function) {}; 	
    ErrorRecoverable(int code, const std::string& message, const char* file = "", size_t line = 0, const char* function = "") :
    	ErrorBase(ErrorBase::ERROR_RECOVERABLE, code, message, file, line, function) {}
    ErrorRecoverable(int code, const char* file = "", size_t line = 0, const char* function = "") :
    	ErrorBase(ErrorBase::ERROR_RECOVERABLE, code, file, line, function) {}

    virtual ~ErrorRecoverable() throw() {}

public:     //// Usage

    virtual std::string as_string() const 
    {
    	c_type<int> code(_code);
    	return ("RECOVERABLE ERROR: " + where() + ": " + std::string(what()) + " [code: " + std::string(code) +  "]");
    }
};

#define THROW_FATAL(message) \
	throw ErrorFatal(message, __FILE__, __LINE__, __PRETTY_FUNCTION__);

#define THROW_RECOVERABLE(message) \
	throw ErrorRecoverable(message, __FILE__, __LINE__, __PRETTY_FUNCTION__);

};

#endif // __TPL_EXCEPTIONS_HH__
