/*
 * Module: rl_nat.cc
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) 2005, 2006, 2007 Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Michael Larson
 * Date: 2005
 * Description:
 *
 * **** End License ****
 *
 */
#include <stdio.h>
#include <iostream> 
#include <fstream>
#include <sstream>
#include <stdlib.h>
#include <string>
#include "rl_nat.hh"

using namespace std;

/**
 *
 **/
NATData::NATData() : 
  _src_port_start(0), 
  _src_port_stop(0), 
  _dst_port_start(0), 
  _dst_port_stop(0),
  _is_new(true)
{
}

/**
 * Generates the correct command string for nat
 **/
XrlCmdError
NATData::create_rule(string &rule_spec) const
{
  if (_type.empty() == true) {
    return XrlCmdError::COMMAND_FAILED("type is not specified\n");
  }

  if (strcasecmp(_type.c_str(), "source") == 0) {
    rule_spec = "-A POSTROUTING ";
    if (strcasecmp(_trans_type.c_str(), "masquerade") == 0) {
      rule_spec += "-j MASQUERADE ";
    }
    else {
      rule_spec += "-j SNAT ";
    }
    if (_out_interface.empty() == false) {
      rule_spec += "-o " + _out_interface + " ";
    } 
    else {
      return XrlCmdError::COMMAND_FAILED("outbound-interface not specified\n");
    }

    if (_out_address.is_zero() == true && 
	_out_network.is_valid() == false && 
	_out_address_start.is_zero() == true && 
	_out_address_stop.is_zero() == true &&
	(strcasecmp(_trans_type.c_str(), "masquerade") != 0)) {
      return XrlCmdError::COMMAND_FAILED("outside-address, outside-network or range not specified\n");
    }

    if (_out_address.is_zero() == false) {
      rule_spec += "--to-source " + _out_address.str() + " ";
    }
    if (_out_network.is_valid() == true) {
      rule_spec += "--to-source " + _out_network.masked_addr().str() + "-" + _out_network.top_addr().str() + " ";
    }

    if (_out_address_start.is_zero() == false && _out_address_stop.is_zero() == false) {
      rule_spec += "--to-source " + _out_address_start.str() + "-" + _out_address_stop.str() + " ";
    }

    if (_dst_address.is_zero() == true && _dst_network.is_valid() == false) {
      return XrlCmdError::COMMAND_FAILED("destination address or network not specified\n");
    }
  }
  else {
    rule_spec += "-A PREROUTING -j DNAT ";
    if (_in_interface.empty() == false) {
      rule_spec += "-i " + _in_interface + " ";
    }
    else {
      return XrlCmdError::COMMAND_FAILED("inbound-interface not specified\n");
    }

    if (_in_address.is_zero() == true && _in_network.is_valid() == false) {
      return XrlCmdError::COMMAND_FAILED("inside-address or inside-network not specified\n");
    }

    if (_in_address.is_zero() == false) {
      rule_spec += "--to-destination " + _in_address.str() + " ";
    }
    if (_in_network.is_valid() == true) {
      rule_spec += "--to-destination " + _in_network.str() + " ";
    }
  }

  if (_protocol.empty() == false) {
    rule_spec += "-p " + _protocol + " ";
  }

  int ct = _src_ports.size() + _src_port_apps.size() + 
    (_src_port_stop -_src_port_start > 0 ? _src_port_stop - _src_port_start : 0);
  if (ct > 0 && (_protocol.empty() || _protocol == "all" || _protocol == "icmp") == true) {
    return XrlCmdError::COMMAND_FAILED("TCP/UDP Protocol must be specified\n");
  }
  if (ct > 1) {
    rule_spec += "-m multiport ";
  } 
  if (ct > 0) {
    rule_spec += "--sport ";
  }
  
  bool first_time = true;
  char buf[80];
  ConstPortIter port_iter = _src_ports.begin();
  while (port_iter != _src_ports.end()) {
    if (first_time == false) {
      rule_spec += ",";
    }
    first_time = false;
    sprintf(buf, "%d", *port_iter);
    rule_spec += string(buf);
    ++port_iter;
  }
  
  ConstAppIter app_iter = _src_port_apps.begin();
  while (app_iter != _src_port_apps.end()) {
    if (first_time == false) {
      rule_spec += ",";
    }
    first_time = false;
    rule_spec += *app_iter;
    ++app_iter;
  }
  
  if (_src_port_stop - _src_port_start > 0) {
    if (first_time == false) {
      rule_spec += ",";
    }
    sprintf(buf, "%d:%d", _src_port_start, _src_port_stop);
    rule_spec += string(buf);
  }
  rule_spec += " ";
  
  ct = _dst_ports.size() + _dst_port_apps.size() + 
    (_dst_port_stop - _dst_port_start > 0 ? _dst_port_stop - _dst_port_start : 0);
  if (ct > 0 && (_protocol.empty() || _protocol == "all" || _protocol == "icmp") == true) {
    return XrlCmdError::COMMAND_FAILED("TCP/UDP Protocol must be specified\n");
  }
  if (ct > 1) {
    rule_spec += "-m multiport ";
  } 
  if (ct > 0) {
    rule_spec += "--dport ";
  }
  
  first_time = true;
  port_iter = _dst_ports.begin();
  while (port_iter != _dst_ports.end()) {
    if (first_time == false) {
      rule_spec += ",";
    }
    first_time = false;
    sprintf(buf, "%d", *port_iter);
    rule_spec += string(buf);
    ++port_iter;
  }
  
  app_iter = _dst_port_apps.begin();
  while (app_iter != _dst_port_apps.end()) {
    if (first_time == false) {
      rule_spec += ",";
    }
    first_time = false;
    rule_spec += *app_iter;
    ++app_iter;
  }
  
  if (_dst_port_stop - _dst_port_start > 0) {
    if (first_time == false) {
      rule_spec += ",";
    }
    sprintf(buf, "%d:%d", _dst_port_start, _dst_port_stop);
    rule_spec += string(buf);
  }
  rule_spec += " ";

  if (_src_address.is_zero() == false) {
    rule_spec += "-s " + _src_address.str() + " ";
  }
  if (_src_network.is_valid() == true) {
    rule_spec += "-s " + _src_network.str() + " ";
  }

  if (_dst_address.is_zero() == false) {
    rule_spec += "-d " + _dst_address.str() + " ";
  }
  if (_dst_network.is_valid() == true) {
    rule_spec += "-d " + _dst_network.str() + " ";
  }

  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
string
NATData::dump() const
{
  string rule = "[" + _type + ", " + _trans_type + ", " + _in_interface + ", " + _out_interface + ", " + _src_address.str() + ", " + _src_network.str() + ", " + _in_address.str() + ", " + _in_network.str() + ", " + _out_address.str() + ", " + _out_network.str() + ", " +  _out_address_start.str() + ", " + _out_address_stop.str() + "]";

  return rule;
}


