/*
 * Module: rl_fileaccess.cc
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) 2005, 2006, 2007 Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Michael Larson
 * Date: 2005
 * Description:
 *
 * **** End License ****
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <string>

#include "rl_fileaccess.hh"

using namespace std;

int rl_fileaccess::MAX_FILENAME_LENGTH = 255;

/**
 *
 */
rl_fileaccess::rl_fileaccess(const std::string &filename) : 
  _in_filename(filename),
  _out_filename(filename),
  _fd_writestream(NULL),
  _fd_readstream(NULL)
{
}

/**
 *
 */
rl_fileaccess::rl_fileaccess(const std::string &in_filename, const std::string &out_filename) : 
  _in_filename(in_filename),
  _out_filename(out_filename),
  _fd_writestream(NULL),
  _fd_readstream(NULL)
{
}

/**
 *
 */
rl_fileaccess::~rl_fileaccess()
{
  fflush(_fd_writestream);

  if (_fd_write < 0 ||
      _fd_readstream == NULL || 
      _fd_writestream == NULL) {
    return;
  }

  close(_fd_write);
  fclose(_fd_readstream);
  fclose(_fd_writestream);

  string filelock(_out_filename + ".lck");
  rename(filelock.c_str(), _out_filename.c_str());
}

/**
 *
 */
void
rl_fileaccess::init(const std::string &filename)
{
  string filelock(filename + ".lck");
  unlink(filelock.c_str());
}

/**
 * get()
 *
 * This method creates a .lck file to prevent concurrent access. 
 * is renamed to replace the active file and the .lck file is removed.
 */
bool
rl_fileaccess::get(FILE*& rd_fd, FILE*& wr_fd)
{
  rd_fd = wr_fd = NULL;

  if ((int)_in_filename.length() > MAX_FILENAME_LENGTH ||
      (int)_out_filename.length() > MAX_FILENAME_LENGTH) {
    return false;
  }

  string out_filelock = _out_filename + ".lck";
  _fd_write = open(out_filelock.c_str(), O_WRONLY | O_CREAT | O_EXCL, 0644);
  if (_fd_write < 0 && errno == EEXIST) {
    return false;
  }
  else if (_fd_write < 0) {
    return false;
  }

  _fd_writestream = fdopen(_fd_write, "w");
  if (_fd_writestream == NULL) {
    close(_fd_write);
    unlink(out_filelock.c_str());
    return false;
  }

  //we are free to lock this file now
  _fd_readstream = fopen(_in_filename.c_str(), "r"); //let's get a file descriptor
  if (_fd_readstream == NULL && errno == ENOENT) {
    //Let's create the file here and try again.
    _fd_readstream = fopen(_out_filename.c_str(), "w+");
    if (_fd_readstream != NULL) {
      fclose(_fd_readstream);
    }
    _fd_readstream = fopen(_in_filename.c_str(), "r"); //let's get a file descriptor
    if (_fd_readstream == NULL) {
      close(_fd_write);
      unlink(out_filelock.c_str());
      return false;
    }
  }
  else if (_fd_readstream == NULL) {
    close(_fd_write);
    unlink(out_filelock.c_str());
    return false;
  }

  rd_fd = _fd_readstream;
  wr_fd = _fd_writestream;
  return true;
}
