#include <iostream>
#include <stdio.h>
#include <string>
#include <map>

#include <linux/types.h>
#include <sys/socket.h>
#include <linux/rtnetlink.h>
#include <linux/netlink.h>
#include "netlink_send.hh"
#include "netlink_listener.hh"
#include "netlink_event.hh"

#ifndef IF_OPER_DORMANT
enum {
	IF_OPER_UNKNOWN,
	IF_OPER_NOTPRESENT,
	IF_OPER_DOWN,
	IF_OPER_LOWERLAYERDOWN,
	IF_OPER_TESTING,
	IF_OPER_DORMANT,
	IF_OPER_UP,
};
#endif


using namespace std;


struct IData
{
public:
  IData() :
    _enabled(false),
    _running(false),
    _mask(0) {}

  bool _enabled;
  bool _running;
  string _iface;
  IPv4 _addr;
  int  _mask;
  int  _if_index;
};

//interface tracking structure
typedef map<string, IData> InterfaceColl;
typedef map<string, IData>::iterator InterfaceIter;

InterfaceColl g_interface;

void flip(int sock, IData &data, bool verbose);


/**
 *
 *
 **/
void 
usage()
{
  fprintf(stdout, "-v   verbose mode\n");
  fprintf(stdout, "-h   help message\n");
}

/**
 *
 *
 **/
int 
main(int argc, char* const argv[])
{
  int ch;
  bool verbose = false;
  while ((ch = getopt(argc, argv, "vh")) != -1) {
    switch (ch) {
    case 'v':
      verbose = true;
      break;
    case 'h':
      usage();
      exit(0);
    }
  }  
  
  if (verbose)
    cout << "nl_daemon()" << endl;
  
  NetlinkSend nl_send;
  NetlinkListener nl_listener;

  int sock = nl_listener.init();
  if (sock <= 0) {
    cerr << "test_netlink(), bad voodoo. exiting.." << endl;
    exit(1);
  }

  if (verbose) {
    cout << "sending initial netlink request" << endl;
  }

  nl_listener.set_multipart(true);
  if (nl_send.send(sock, RTM_GETLINK) != 0) {
    cerr << "test_netlink(), error sending" << endl;
    exit(1);
  }

  bool send_request = true;
  while (true) {
    NetlinkEvent nl_event;
    if (nl_listener.process(nl_event) == true) {
      if (verbose) {
	cout << "received msg from kernel" << endl;
      }

      if (send_request) {
	if (nl_send.send(sock, RTM_GETADDR) != 0) {
	  cerr << "test_netlink(), error sending" << endl;
	  exit(1);
	}
	send_request = false;
      }
      else {
	nl_listener.set_multipart(false);
      }


      //now track interface state change
      string iface = nl_event.get_iface();
      if (iface.empty()) {
	continue;
      }
      if (nl_event.get_type() == RTM_NEWLINK) {
	InterfaceIter iter = g_interface.find(iface);
	if (iter != g_interface.end()) {
	  if (iter->second._running != nl_event.get_running()) {
	    //state changed!
	    if (verbose) {
	      cout << "state changed for: " << iface << endl;
	    }
	    iter->second._iface = iface;
	    iter->second._running = nl_event.get_running();
	    iter->second._enabled = nl_event.get_enabled();
	    iter->second._if_index = nl_event.get_index();
	    flip(sock, iter->second, verbose);
	  }
	}
	else {
	  if (verbose) {
	    cout << "found interface: " << iface << endl;
	  }
	  IData data;
	  data._enabled = nl_event.get_enabled();
	  data._running = nl_event.get_running();
	  data._if_index = nl_event.get_index();
	  g_interface.insert(pair<string,IData>(iface,data));
	}
      }
      else if (nl_event.get_type() == RTM_DELLINK) {
      }
      else if (nl_event.get_type() == RTM_NEWADDR) {
	InterfaceIter iter = g_interface.find(iface);
	if (iter == g_interface.end()) {
	  char buf[40];
	  sprintf(buf, "%d", g_interface.size());
	  cerr << "ERROR: interface events out of order: " << iface << ", coll size: " << string(buf) << endl;
	  break;
	}
	iter->second._addr = nl_event.get_addr();
	iter->second._mask = nl_event.get_mask_len();
      }
      else if (nl_event.get_type() == RTM_DELADDR) {
      }


      if (verbose) {
	char buf[20];
	sprintf(buf, "%d", nl_event.get_index());
	cout << "results for " << nl_event.get_iface() << "(" << string(buf) << ")" << endl;
	if (nl_event.get_type() == RTM_DELLINK || 
	    nl_event.get_type() == RTM_NEWLINK) {
	  cout << "  type: " << string(nl_event.get_type()==RTM_DELLINK?"DELLINK":"NEWLINK") << endl;
	  cout << "  enabled: " << string(nl_event.get_enabled()?"UP":"DOWN") << endl;
	  cout << "  running: " << string(nl_event.get_running()?"UP":"DOWN") << endl;
	  cout << "  mac: " << nl_event.get_mac_str() << endl;
	}
	cout << endl;
      }

    }
    else {
      //      cout << "didn't receive a message, sleeping for 1 second" << endl;
      cout << "heartbeat" << endl;
      sleep(1);
    }
  }
  exit(0);
}


void flip(int fd, IData &data, bool verbose)
{
  struct {
    struct nlmsghdr hdr;
      struct ifinfomsg ifinfo;
    char opts[16];
  } req;
  memset(&req, 0x0, sizeof(req));
    

  req.hdr.nlmsg_len = NLMSG_LENGTH(sizeof(struct ifinfomsg));
  req.hdr.nlmsg_type = RTM_SETLINK;
  req.hdr.nlmsg_flags = NLM_F_REQUEST;
  req.hdr.nlmsg_seq = time(NULL);
  req.hdr.nlmsg_pid = 0;

  req.ifinfo.ifi_family = AF_UNSPEC;
  req.ifinfo.ifi_type = 0;
  req.ifinfo.ifi_index = data._if_index;
  req.ifinfo.ifi_flags = 0;
  req.ifinfo.ifi_change = 0;

  struct rtattr *rta;
  rta = (struct rtattr *)
    ((char *)&req + NLMSG_ALIGN(req.hdr.nlmsg_len));
  rta->rta_type = IFLA_OPERSTATE;
  rta->rta_len = RTA_LENGTH(sizeof(char));
  *((char *) RTA_DATA(rta)) = !data._running ? IF_OPER_DORMANT : IF_OPER_UP;
  req.hdr.nlmsg_len = NLMSG_ALIGN(req.hdr.nlmsg_len) +
    RTA_LENGTH(sizeof(char));
  
  if (send(fd, &req, req.hdr.nlmsg_len, 0) < 0) {
    close(fd);
    cerr << "error in sendto" << endl;
    exit(1);
  }

  if (verbose) {
    cout << "sending message to interface: " << data._iface << "(" << data._if_index << ") " << string(!data._running ? "IF_OPER_DORMANT" : "IF_OPER_UP") << endl;
  }

  data._enabled = data._running = !data._running;

  return;
}
