// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-

// Copyright (c) 2001-2006 International Computer Science Institute
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software")
// to deal in the Software without restriction, subject to the conditions
// listed in the XORP LICENSE file. These conditions include: you must
// preserve this copyright notice, and you cannot mention the copyright
// holders in advertising related to the Software without their permission.
// The Software is provided WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED. This
// notice is a summary of the XORP LICENSE file; the license in that file is
// legally binding.

#ident "$XORP: xorp/libfeaclient/ifmgr_xrl_mirror.cc,v 1.19 2006/03/16 00:04:11 pavlin Exp $"

#include "libxorp/status_codes.h"
#include "libxorp/eventloop.hh"

#include "libxipc/xrl_std_router.hh"
#include "xrl/interfaces/fea_ifmgr_replicator_xif.hh"
#include "ifmgr_cmds.hh"
#include "ifmgr_xrl_observer.hh"
#include "ifmgr_xrl_mirror_target.hh"
#include "ifmgr_xrl_mirror.hh"

#include <syslog.h>

// ----------------------------------------------------------------------------
// IfMgrXrlMirrorRouter
//
// Class to act as an XrlRouter for IfMgrXrlMirror
//

class IfMgrXrlMirrorRouter : public XrlStdRouter
{
public:
    IfMgrXrlMirrorRouter(EventLoop&	e,
			 const char*	class_name)
	: XrlStdRouter(e, class_name), _o(0)
    {}

    IfMgrXrlMirrorRouter(EventLoop&	e,
			 const char*	class_name,
			 IPv4		finder_addr)
	: XrlStdRouter(e, class_name, finder_addr), _o(0)
    {}

    IfMgrXrlMirrorRouter(EventLoop&	e,
			 const char*	class_name,
			 IPv4		finder_addr,
			 uint16_t	finder_port)
	: XrlStdRouter(e, class_name, finder_addr, finder_port), _o(0)
    {}

    IfMgrXrlMirrorRouter(EventLoop&	e,
			 const char*	class_name,
			 const char*	finder_hostname,
			 uint16_t	finder_port)
	: XrlStdRouter(e, class_name, finder_hostname, finder_port), _o(0)
    {}

    /**
     * Attach an observer to router.  Only one observer is permitted.
     *
     * @param o observer to attach.
     * @return true on success, false if an observer is already registered;
     */
    bool attach(IfMgrXrlMirrorRouterObserver* o)
    {
	if (_o)
	    return false;
	_o = o;
	return true;
    }

    /**
     * Detach an observer to router.
     *
     * @param o observer to detach.
     * @return true on success, false if another observer is attached;
     */
    bool detach(IfMgrXrlMirrorRouterObserver* o)
    {
	if (o != _o)
	    return false;
	_o = 0;
	return true;
    }

protected:
    void finder_ready_event(const string& tgt_name)
    {
	if (tgt_name == instance_name() && _o != 0) {
	    _o->finder_ready_event();
	}
    }

    void finder_disconnect_event()
    {
	if (_o != 0) {
	    _o->finder_disconnect_event();
	}
    }

protected:
    IfMgrXrlMirrorRouterObserver* _o;
};


// ----------------------------------------------------------------------------
// IfMgrXrlMirror

static const char* CLSNAME = "ifmgr_mirror";

IfMgrXrlMirror::IfMgrXrlMirror(EventLoop&	e,
			       const char*	rtarget,
			       IPv4		finder_addr,
			       uint16_t		finder_port)

    : ServiceBase("FEA Interface Mirror"),
      _e(e), _finder_addr(finder_addr), _finder_port(finder_port),
      _dispatcher(_iftree), _rtarget(rtarget), _rtr(0), _xrl_tgt(0), _nl(e)
{
}

IfMgrXrlMirror::IfMgrXrlMirror(EventLoop&	e,
			       const char*	rtarget,
			       const char*	finder_hostname,
			       uint16_t		finder_port)

    : ServiceBase("FEA Interface Mirror"),
      _e(e), _finder_hostname(finder_hostname), _finder_port(finder_port),
      _dispatcher(_iftree), _rtarget(rtarget), _rtr(0), _xrl_tgt(0), _nl(e)
{
}

IfMgrXrlMirror::~IfMgrXrlMirror()
{
    if (_rtr != NULL) {
	//
	// Both _rtr and _xrl_tgt have been assigned in startup() so
	// if one is NULL then the other must be NULL and we need to
	// clean-up.  We test in case startup() has not been called.
	//
	_xrl_tgt->detach(this);
	_rtr->detach(this);
	delete _xrl_tgt;
	_xrl_tgt = 0;
	delete _rtr;
	_rtr = 0;
    }
}

bool
IfMgrXrlMirror::startup()
{

    if (status() != SERVICE_READY)
	return false;

    if (_rtr == NULL) {
	if (! _finder_hostname.empty()) {
	    _rtr = new IfMgrXrlMirrorRouter(_e, CLSNAME,
					    _finder_hostname.c_str(),
					    _finder_port);
	} else {
	    _rtr = new IfMgrXrlMirrorRouter(_e, CLSNAME,
					    _finder_addr,
					    _finder_port);
	}
	_rtr->attach(this);
    }
    if (_xrl_tgt == NULL) {
	_xrl_tgt = new IfMgrXrlMirrorTarget(*_rtr, _dispatcher);
	_xrl_tgt->attach(this);
    }

    string err;

    if (_nl.start(err, _xrl_tgt) != XORP_OK) {
	syslog(LOG_USER | LOG_ERR, "IfMgrXrlMirror::start(), error on initializing netlink_head, exiting.");
	cerr << "err: " << err << endl;
	return false;
    }

    set_status(SERVICE_STARTING, "Initializing Xrl Router.");

    return true;
}

bool
IfMgrXrlMirror::shutdown()
{
    if (status() != SERVICE_RUNNING)
	return false;

    set_status(SERVICE_SHUTTING_DOWN);

    string err;
    if (_nl.stop(err) != XORP_OK) {
	syslog(LOG_USER | LOG_ERR, "IfMgrXrlMirror::stop(), error on finalizing netlink_head, exiting.");
	cerr << "err: " << err << endl;
	return false;
    }

    unregister_with_ifmgr();

    return true;
}

void
IfMgrXrlMirror::finder_disconnect_event()
{
    _iftree.clear();
    if (status() == SERVICE_SHUTTING_DOWN) {
	set_status(SERVICE_SHUTDOWN);
    } else {
	set_status(SERVICE_FAILED);
    }
}

void
IfMgrXrlMirror::finder_ready_event()
{
        register_with_ifmgr();
}

void
IfMgrXrlMirror::tree_complete()
{
    if (status() != SERVICE_STARTING) {
	// XXX In case shutdown is called or a failure occurs before
	// tree_complete message is received.  If either happens we don't
	// want to advertise the tree as complete.
	return;
    }

    set_status(SERVICE_RUNNING);
    list<IfMgrHintObserver*>::const_iterator ci;
    for (ci = _hint_observers.begin(); ci != _hint_observers.end(); ++ci) {
	IfMgrHintObserver* ho = *ci;
	ho->tree_complete();
    }
}

void
IfMgrXrlMirror::updates_made()
{
    if (status() & ServiceStatus(SERVICE_SHUTTING_DOWN
				 | SERVICE_SHUTDOWN
				 | SERVICE_FAILED))
	return;

    list<IfMgrHintObserver*>::const_iterator ci;
    for (ci = _hint_observers.begin(); ci != _hint_observers.end(); ++ci) {
	IfMgrHintObserver* ho = *ci;
	ho->updates_made();
    }
}

bool
IfMgrXrlMirror::attach_hint_observer(IfMgrHintObserver* o)
{
    if (status() & ServiceStatus(SERVICE_SHUTTING_DOWN
				 | SERVICE_SHUTDOWN
				 | SERVICE_FAILED))
	return false;

    if (find(_hint_observers.begin(), _hint_observers.end(), o)
	!= _hint_observers.end()) {
	return false;
    }
    _hint_observers.push_back(o);
    return true;
}

bool
IfMgrXrlMirror::detach_hint_observer(IfMgrHintObserver* o)
{
    list<IfMgrHintObserver*>::iterator i;
    i = find(_hint_observers.begin(), _hint_observers.end(), o);
    if (i == _hint_observers.end()) {
	return false;
    }
    _hint_observers.erase(i);
    return true;
}

void
IfMgrXrlMirror::register_with_ifmgr()
{
    XrlIfmgrReplicatorV0p1Client c(_rtr);
    if (c.send_register_ifmgr_mirror(_rtarget.c_str(),
				     _rtr->instance_name(),
				     callback(this,
					      &IfMgrXrlMirror::register_cb))
	== false) {
	set_status(SERVICE_FAILED, "Failed to send registration to ifmgr");
	return;
    }
    set_status(SERVICE_STARTING, "Registering with FEA interface manager.");
}

void
IfMgrXrlMirror::register_cb(const XrlError& e)
{
    if (e == XrlError::OKAY()) {
	set_status(SERVICE_STARTING, "Waiting to receive interface data.");
    } else {
	set_status(SERVICE_FAILED, "Failed to send registration to ifmgr");
    }
}

void
IfMgrXrlMirror::unregister_with_ifmgr()
{
    XrlIfmgrReplicatorV0p1Client c(_rtr);
    if (c.send_unregister_ifmgr_mirror(_rtarget.c_str(), _rtr->instance_name(),
		callback(this, &IfMgrXrlMirror::unregister_cb))
	== false) {
	set_status(SERVICE_FAILED, "Failed to send unregister to FEA");
	return;
    }
    set_status(SERVICE_SHUTTING_DOWN,
	       "De-registering with FEA interface manager.");
}

void
IfMgrXrlMirror::unregister_cb(const XrlError& e)
{
    _iftree.clear();

    if (e == XrlError::OKAY()) {
	set_status(SERVICE_SHUTDOWN);
    } else {
	set_status(SERVICE_FAILED, "Failed to de-registration to ifmgr");
    }
}
