// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-

// Copyright (c) 2001-2007 International Computer Science Institute
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software")
// to deal in the Software without restriction, subject to the conditions
// listed in the XORP LICENSE file. These conditions include: you must
// preserve this copyright notice, and you cannot mention the copyright
// holders in advertising related to the Software without their permission.
// The Software is provided WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED. This
// notice is a summary of the XORP LICENSE file; the license in that file is
// legally binding.

#ident "$XORP$"

#include "print_routes.hh"
#include "bgp/aspath.hh"
#include "bgp/path_attribute.hh"

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#if defined(HAVE_REGEX_H)
#include <regex.h>
#elif defined(HAVE_PCREPOSIX_H)
#include <pcre.h>
#include <pcreposix.h>
#else
#error "No header defining regex_t is present."
#endif


bool _terminated = false;
PrintRoutesBase *_route_printer = NULL;

void
terminate_main_loop(int sig)
{
    debug_msg("Signal %d\n", sig);
    UNUSED(sig);

    if (_route_printer != NULL)
	_route_printer->terminate();
    _terminated = true;
}

struct SignalGuard {
    SignalGuard(PrintRoutesBase * route_printer) {
	_route_printer = route_printer;
	_old_handler = signal(SIGINT, terminate_main_loop);
	_old_handler = signal(SIGTERM, terminate_main_loop);
	_old_handler = signal(SIGPIPE, terminate_main_loop);
    }

    ~SignalGuard() {
	signal(SIGINT, _old_handler);
	signal(SIGTERM, _old_handler);
	signal(SIGPIPE, _old_handler);
	_route_printer = NULL;
    }
private:
    sighandler_t _old_handler;
};

void usage()
{
    fprintf(stderr,
//	    "Usage:\n"
//	    "  xorpsh_print_routes [options] show bgp routes [arguments]\n"
	    "  show bgp routes [arguments]\n"
	    "  show bgp received-routes [arguments]\n"
	    "  show bgp dampened-routes [arguments]\n"
	    "arguments:\n"
	    "  [ipv4|ipv6]          - print only IPv4 or IPv6 routes (default: both)\n"
	    "  [unicast|multicast]  - print only unicast or multicust routes (default: both)\n"
	    "  [summary|detail]     - print detailed route attributes or brief output (default: summary)\n"
	    "  [peer x.x.x.x]       - print routes originated by specified peer (default: full table)\n"
	    "  [prefix x.x.x.x/n]   - print matching routes for specified prefix (default: full table)\n"
	    "  [address x.x.x.x]    - print matching routes for specified address (default: full table)\n"
	    "  [aspath|community <regex>]  - print matching routes for specified address (default: full table)\n");
}

int main(int argc, char **argv)
{
    XorpUnexpectedHandler x(xorp_unexpected_handler);
    //
    // Initialize and start xlog
    //
    xlog_init(argv[0], NULL);
    xlog_set_verbose(XLOG_VERBOSE_LOW);		// Least verbose messages
    // XXX: verbosity of the error messages temporary increased
    xlog_level_set_verbose(XLOG_LEVEL_ERROR, XLOG_VERBOSE_HIGH);
    xlog_add_default_output();
    xlog_start();

    IPNet<IPv4> net_ipv4;
    IPNet<IPv6> net_ipv6;
    IPv4 peer;
    string regex;
    int field = 0;
    RouteTableType table_type;
    bool ipv4, ipv6, unicast, multicast, bulk, file;
    ipv4 = ipv6 = unicast = multicast = bulk = file = false;

    PrintRoutes<IPv4>::detail_t verbose_ipv4 = PrintRoutes<IPv4>::NORMAL;
    PrintRoutes<IPv6>::detail_t verbose_ipv6 = PrintRoutes<IPv6>::NORMAL;
    int interval = -1;
    int lines = -1;
    int records = -1;
    int window = -1;

    int c;
    while ((c = getopt(argc, argv, "46umbfl:r:w:")) != -1) {
	switch (c) {
	case '4':
	    ipv4 = true;
	    break;
	case '6':
	    ipv6 = true;
	    break;
	case 'u':
	    unicast = true;
	    break;
	case 'm':
	    multicast = true;
	    break;
	case 'b':
	    bulk = true;
	    break;
	case 'f':
	    file = true;
	    break;
	case 'l':
	    lines = atoi(optarg);
	    break;
	case 'r':
	    records = atoi(optarg);
	    break;
	case 'w':
	    window = atoi(optarg);
	    break;
	default:
	    printf("hello\n");
	    usage();
	    return -1;
	}
    }

    argc -= optind;
    argv += optind;

    if ((argc < 3) || (argc > 9)) {
	usage();
	return -1;
    }

    if (strcmp(argv[0], "show") != 0) {
	usage();
	return -1;
    }
    if (strcmp(argv[1], "bgp") != 0) {
	usage();
	return -1;
    }
    if (strcmp(argv[2], "routes") == 0) {
	table_type = DECISION_TABLE;
    } else if (strcmp(argv[2], "neighbor-routes") == 0) {
	table_type = RIB_IN_TABLE;
    } else if (strcmp(argv[2], "dampened-routes") == 0) {
	table_type = DAMPING_TABLE;
    } else if (strcmp(argv[2], "advertised-routes") == 0) {
	table_type = DAMPING_TABLE;
    }
    else {
	usage();
	return -1;
    }
    int counter = 4;
    if (argc > 3) {
	if (strcmp(argv[3], "ipv4") == 0) {
	    ipv4 = true;
	    counter++;
	}
	if (strcmp(argv[3], "ipv6") == 0) {
	    ipv6 = true;
	    counter++;
	}
    }
    if (argc > 4) {
	if (strcmp(argv[4], "unicast") == 0) {
	    unicast = true;
	    counter++;
	}
	if (strcmp(argv[4], "multicast") == 0) {
	    multicast = true;
	    counter++;
	}
    }
    if (argc >= counter) {
	if (strcmp(argv[counter - 1], "summary")==0) {
	    verbose_ipv4 = PrintRoutes<IPv4>::SUMMARY;
	    verbose_ipv6 = PrintRoutes<IPv6>::SUMMARY;
	    counter++;
	} else if (strcmp(argv[counter - 1], "detail")==0) {
	    verbose_ipv4 = PrintRoutes<IPv4>::DETAIL;
	    verbose_ipv6 = PrintRoutes<IPv6>::DETAIL;
	    counter++;
	}
    }
    if (argc >= (counter + 1) &&
	strcmp(argv[counter - 1], "peer") == 0) {
	    try {
		IPv4 ip(argv[counter]);
		peer = ip;
		counter += 2;
	    }
	    catch(...) {
		fprintf(stderr, "Invalid BGP ID specified: %s\n", argv[counter]);
		return -1;
	    }
    }
    if (argc == counter + 1) {
	if (strcmp(argv[counter - 1], "prefix") == 0) {
	    try {
		IPNet<IPv4> subnet(argv[counter]);
		net_ipv4 = subnet;
	    } catch(...) {
		try {
		    IPNet<IPv6> subnet(argv[counter]);
		    net_ipv6 = subnet;
		} catch(...) {
		    // xorp_catch_standard_exceptions();
		    fprintf(stderr, "Invalid network prefix specified: %s\n", argv[counter]);
		    return -1;
		}
	    }
	}
	else if (strcmp(argv[counter - 1], "address") == 0) {
	    try {
		IPv4 addr(argv[counter]);
		net_ipv4 = IPNet<IPv4>(addr, 32);
	    } catch(...) {
		try {
		    IPv6 addr(argv[counter]);
		    net_ipv6 = IPNet<IPv6>(addr, 128);
		} catch(...) {
		    // xorp_catch_standard_exceptions();
		    fprintf(stderr, "Invalid IP address specified: %s\n", argv[counter]);
		    return -1;
		}
	    }
	}
	else if (strcmp(argv[counter - 1], "aspath") == 0) {
	    regex = string(argv[counter]);
	    field = 1;  // FIELD_ASPATH
	    regex_t *re = new regex_t;
	    int res = regcomp(re, regex.c_str(), REG_EXTENDED);
	    if (res) {
		// xorp_catch_standard_exceptions();
		fprintf(stderr, "Invalid regular expression: %s\n", argv[counter]);
		return -1;
	    }
	    regfree(re);
	    delete re;
	}
	else if (strcmp(argv[counter - 1], "community") == 0) {
	    regex = string(argv[counter]);
	    field = 2;  // FIELD_COMMUNITY
	    regex_t *re = new regex_t;
	    int res = regcomp(re, regex.c_str(), REG_EXTENDED);
	    if (res) {
		// xorp_catch_standard_exceptions();
		fprintf(stderr, "Invalid regular expression: %s\n", argv[counter]);
		return -1;
	    }
	    regfree(re);
	    delete re;
	}
    }

    if ((ipv4 == false) && (ipv6 == false))
	ipv4 = true;

    if ((unicast == false) && (multicast == false))
	unicast = true;

    TransferMode mode = SINGLE_REQUEST;
    if (bulk)
	mode = BULK_REQUEST;
    if (file)
	mode = FILE_REQUEST;

    try {
	if (ipv4) {
	    PrintRoutes<IPv4> route4_printer(verbose_ipv4, interval, net_ipv4, peer,
					    unicast, multicast, regex, field, table_type, mode, lines);
	    if (records > 0)
		route4_printer.set_max_records(records);

	    if (window > 0)
		route4_printer.set_max_requests(window);

	    SignalGuard guard(&route4_printer);
	    route4_printer.run();
	}
	if (!_terminated && ipv6) {
	    PrintRoutes<IPv6> route6_printer(verbose_ipv6, interval, net_ipv6, peer,
					    unicast, multicast, regex, field, table_type, mode, lines);
	    if (records > 0)
		route6_printer.set_max_records(records);

	    if (window > 0)
		route6_printer.set_max_requests(window);

	    SignalGuard guard(&route6_printer);
	    route6_printer.run();
	}
    } catch(...) {
	xorp_catch_standard_exceptions();
    }

    //
    // Gracefully stop and exit xlog
    //
    xlog_stop();
    xlog_exit();
    return 0;
}

