// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-

// Copyright (c) 2001-2007 International Computer Science Institute
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software")
// to deal in the Software without restriction, subject to the conditions
// listed in the XORP LICENSE file. These conditions include: you must
// preserve this copyright notice, and you cannot mention the copyright
// holders in advertising related to the Software without their permission.
// The Software is provided WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED. This
// notice is a summary of the XORP LICENSE file; the license in that file is
// legally binding.

// $XORP$

#ifndef __BGP_TOOLS_PRINT_PEER_HH__
#define __BGP_TOOLS_PRINT_PEER_HH__

#include "bgptools_module.h"

#include "bgp/aspath.hh"
#include "bgp/path_attribute.hh"
#include "bgp/route_table_base.hh"

#include "libxorp/xorp.h"
#include "libxorp/xlog.h"

#include "libxipc/xrl_router.hh"
#include "libxipc/xrl_args.hh"

#include "xrl/interfaces/bgp_xif.hh"

class PrintRoutesBase {
public:
    virtual ~PrintRoutesBase() = 0;
    virtual void terminate()  = 0;
};

enum TransferMode { SINGLE_REQUEST, BULK_REQUEST, FILE_REQUEST };

template <typename A>
class PrintRoutes : public XrlBgpV0p2Client, public PrintRoutesBase {
public:
    static const uint32_t MAX_REQUESTS = 100;
    static const uint32_t MAX_BULK_REQUESTS = 1;
    static const uint32_t MAX_BULK_RECORDS = 100;
    static const int32_t INVALID = -1;

    static const uint32_t NO_EXPORT = 0xFFFFFF01;  // RFC 1997
    static const uint32_t NO_ADVERTISE = 0xFFFFFF02;  // RFC 1997
    static const uint32_t NO_EXPORT_SUBCONFED = 0xFFFFFF03;  // RFC 1997

    enum detail_t {SUMMARY, NORMAL, DETAIL};
    PrintRoutes(detail_t verbose, int interval, const IPNet<A>& net, const IPv4& peer,
	bool unicast, bool multicast, const string& regex, int field,
	RouteTableType type, TransferMode mode, int lines = -1);
    ~PrintRoutes();
    void get_route_list_start(const IPNet<A>& net, bool unicast, bool multicast);
    void get_route_list_start_done(const XrlError& e,
				   const uint32_t* token);
    void get_route_list_next();
    void get_route_list_next_done(const XrlError& 	 e,
				  const IPv4* 		 peer_id,
				  const IPNet<A>* 	 net,
				  const uint32_t 	 *best_and_origin,
				  const vector<uint8_t>* aspath,
				  const A* 		 nexthop,
				  const int32_t* 	 med,
				  const int32_t* 	 localpref,
				  const int32_t* 	 atomic_agg,
				  const vector<uint8_t>* aggregator,
				  const int32_t* 	 calc_localpref,
				  const vector<uint8_t>* attr_unknown,
				  const bool* 		 valid,
				  const bool* 		 unicast,
				  const bool* 		 multicast);
    void get_route_list_bulk_start(const IPNet<A>& net, const IPv4& peer, bool unicast, bool multicast, RouteTableType type);
    void get_route_list_bulk_start_done(const XrlError& e,
				   const uint32_t* token);
    void get_route_list_bulk_file_next();
    void get_route_list_bulk_file_next_done(const XrlError& 	 e,
				  const string* 	 filename,
				  const bool* 		 valid);
    void get_route_list_bulk_next();
    void get_route_list_bulk_next_done(const XrlError& 	 e,
				  const vector<uint8_t>* data,
				  const bool* 		 valid);
    void set_max_requests(int max_req);
    void set_max_records(int max_rec);
    void run();
    void terminate();
private:
    void print_route(const IPv4& peer_id,
                                  const IPNet<A>& net,
                            	  uint8_t best,
                            	  uint8_t origin,
                            	  const AsPath asp,
                                  const A& nexthop,
                                  const int32_t& med,
                                  const int32_t& localpref,
                                  const int32_t& atomic_agg,
                            	  const vector<uint8_t>& aggregator,
				  const vector<uint8_t>& community);
    void timer_expired();

    EventLoop&	 	_eventloop;
    XrlPFListener*	_l;
    XrlRouter 		_xrl_rtr;
    detail_t 		_verbose;
    IPNet<A>		_net;
    IPv4		_peer;
    uint32_t 		_token;
    bool 		_done;
    uint32_t 		_count;
    bool 		_prev_no_bgp;
    bool 		_prev_no_routes;

    RouteTableType	_type;
    TransferMode	_mode;
    XorpTimer 		_timer;
    int 		_active_requests;
    bool		_use_files;
    bool 		_unicast;
    bool 		_multicast;
    string		_regex;
    int			_field;
    int			_lines;
    int 		_interval;
    uint32_t		_limit;
    uint32_t 		_max_requests;
};

#endif // __BGP_TOOLS_PRINT_PEER_HH__
