// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-

// Copyright (c) 2001-2007 International Computer Science Institute
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software")
// to deal in the Software without restriction, subject to the conditions
// listed in the XORP LICENSE file. These conditions include: you must
// preserve this copyright notice, and you cannot mention the copyright
// holders in advertising related to the Software without their permission.
// The Software is provided WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED. This
// notice is a summary of the XORP LICENSE file; the license in that file is
// legally binding.

#ident "$XORP$"

#include "print_routes.hh"
#include "libxorp/eventloop_factory.hh"
#include "bgp/route_dump_record.hh"
#include "libxipc/xrl_pf_inproc.hh"
#include "libxipc/xrl_pf_stcp.hh"
#include "libxipc/xrl_pf_sudp.hh"
#include <stdio.h>
#include <fcntl.h>

// ----------------------------------------------------------------------------
// Helper methods

static XrlPFListener*
create_listener(EventLoop& e, XrlDispatcher* d)
{
    const char* pf = getenv("XORP_PF");
    if (pf != NULL) {
	if (pf[0] == 'i') {
	    return new XrlPFInProcListener(e, d);
	}
	if (pf[0] == 'u') {
	    return new XrlPFSUDPListener(e, d);
	}
    }
    return new XrlPFSTCPListener(e, d);
}

static void
destroy_listener(XrlPFListener*& l)
{
    delete l;
    l = 0;
}

PrintRoutesBase::~PrintRoutesBase()
{
}

// ----------------------------------------------------------------------------
// Specialized PrintRoutes implementation

template <>
void
PrintRoutes<IPv4>::get_route_list_start(const IPNet<IPv4>& net, bool unicast,
					bool multicast)
{
    _active_requests = 0;
    send_get_v4_route_list_start("bgp", net, unicast, multicast,
		 callback(this, &PrintRoutes::get_route_list_start_done));
}

template <>
void
PrintRoutes<IPv6>::get_route_list_start(const IPNet<IPv6>& net, bool unicast,
					bool multicast)
{
    _active_requests = 0;
    send_get_v6_route_list_start("bgp", net, unicast, multicast,
		 callback(this, &PrintRoutes::get_route_list_start_done));
}

template <>
void
PrintRoutes<IPv4>::get_route_list_next()
{
    send_get_v4_route_list_next("bgp",	_token,
		callback(this, &PrintRoutes::get_route_list_next_done));
}

template <>
void
PrintRoutes<IPv6>::get_route_list_next()
{
    send_get_v6_route_list_next("bgp", _token,
		callback(this, &PrintRoutes::get_route_list_next_done));
}

template <>
void
PrintRoutes<IPv4>::get_route_list_bulk_start(const IPNet<IPv4>& net, const IPv4& peer, bool unicast,
					bool multicast, RouteTableType type)
{
    _active_requests = 0;
    send_get_v4_route_list_bulk_start("bgp", net, peer, unicast, multicast, _regex, _field, type,
		 callback(this, &PrintRoutes<IPv4>::get_route_list_bulk_start_done));
}

template <>
void
PrintRoutes<IPv6>::get_route_list_bulk_start(const IPNet<IPv6>& net, const IPv4& peer, bool unicast,
					bool multicast, RouteTableType type)
{
    _active_requests = 0;
    send_get_v6_route_list_bulk_start("bgp", net, peer, unicast, multicast, _regex, _field, type,
		 callback(this, &PrintRoutes<IPv6>::get_route_list_bulk_start_done));
}

template <>
void
PrintRoutes<IPv4>::get_route_list_bulk_file_next()
{
    send_get_v4_route_list_bulk_file_next("bgp", _token, _limit,
		callback(this, &PrintRoutes<IPv4>::get_route_list_bulk_file_next_done));
}

template <>
void
PrintRoutes<IPv6>::get_route_list_bulk_file_next()
{
    send_get_v6_route_list_bulk_file_next("bgp", _token, _limit,
		callback(this, &PrintRoutes<IPv6>::get_route_list_bulk_file_next_done));
}

template <>
void
PrintRoutes<IPv4>::get_route_list_bulk_next()
{
    send_get_v4_route_list_bulk_next("bgp", _token, _limit,
		callback(this, &PrintRoutes<IPv4>::get_route_list_bulk_next_done));
}

template <>
void
PrintRoutes<IPv6>::get_route_list_bulk_next()
{
    send_get_v6_route_list_bulk_next("bgp", _token, _limit,
		callback(this, &PrintRoutes<IPv6>::get_route_list_bulk_next_done));
}

// ----------------------------------------------------------------------------
// Common PrintRoutes implementation

template <typename A>
PrintRoutes<A>::PrintRoutes(detail_t verbose, int interval, const IPNet<A>& net, const IPv4& peer,
			    bool unicast, bool multicast, const string& regex, int field,
			    RouteTableType type, TransferMode mode, int lines)
    : XrlBgpV0p2Client(&_xrl_rtr), _eventloop(*EventLoopFactory::instance().create(eventloop_st)),
      _xrl_rtr(_eventloop, "print_routes", IPv4("127.0.0.1"),
	FinderConstants::FINDER_DEFAULT_PORT()), _verbose(verbose), _net(net), _peer(peer), _type(type), _mode(mode),
      _unicast(unicast), _multicast(multicast), _regex(regex), _field(field), _lines(lines), _interval(interval)
{
    _l = create_listener(_eventloop, &_xrl_rtr);
    _xrl_rtr.add_listener(_l);
    _prev_no_bgp = false;
    _prev_no_routes = false;
    _limit = MAX_BULK_RECORDS;
    _max_requests = MAX_BULK_REQUESTS;

    // Wait for the finder to become ready.
    {
	bool timed_out = false;
	XorpTimer t = _eventloop.set_flag_after_ms(10000, &timed_out);
	while (_xrl_rtr.connected() == false && timed_out == false) {
	    _eventloop.run();
	}

	if (_xrl_rtr.connected() == false) {
	    XLOG_WARNING("XrlRouter did not become ready. No Finder?");
	}
    }
}

template <typename A>
PrintRoutes<A>::~PrintRoutes()
{
    // remove_listener(&_l);
    destroy_listener(_l);
}

template <typename A>
void
PrintRoutes<A>::set_max_requests(int max_req)
{
    _max_requests = max_req;
}

template <typename A>
void
PrintRoutes<A>::set_max_records(int max_rec)
{
    _limit = max_rec;
}

template <typename A>
void
PrintRoutes<A>::run()
{
    for (;;) {
	_done = false;
	_token = 0;
	_count = 0;
	switch (_mode) {
	case SINGLE_REQUEST:
	    get_route_list_start(_net, _unicast, _multicast);
	    break;
	case BULK_REQUEST:
	case FILE_REQUEST:
	    get_route_list_bulk_start(_net, _peer, _unicast, _multicast, _type);
	    break;
        };

	while (_done == false || _active_requests > 0) {
	    _eventloop.run();
	    if (_lines == static_cast<int>(_count)) {
		printf("Output truncated at %u lines\n",
		       XORP_UINT_CAST(_count));
		break;
	    }
	}
	if (_interval <= 0)
	    break;

	//delay before next call
	XorpCallback0<void>::RefPtr cb
	    = callback(this, &PrintRoutes::timer_expired);
	_done = false;
	_timer = _eventloop.new_oneoff_after_ms(_interval*1000, cb);
	while (_done == false) {
	    _eventloop.run();
	}
    }
}

template <typename A>
void
PrintRoutes<A>::terminate()
{
    _done = true;
}

template <typename A>
void
PrintRoutes<A>::get_route_list_start_done(const XrlError& e,
					  const uint32_t* token)
{
    if (e != XrlError::OKAY()) {
	//fprintf(stderr, "Failed to get peer list start\n");
	if (_prev_no_bgp == false)
	    printf("\n\nBGP is not configured\n");
	_prev_no_bgp = true;
	_done = true;
	return;
    }
    _prev_no_bgp = false;

    switch(_verbose) {
    case SUMMARY:
    case NORMAL:
	printf("Status Codes: * valid route, > best route\n");
	printf("Origin Codes: i IGP, e EGP, ? incomplete\n\n");
	printf("   Prefix                Nexthop                    "
	       "Peer            AS Path\n");
	printf("   ------                -------                    "
	       "----            -------\n");
	break;
    case DETAIL:
	break;
    }

    _token = *token;
    for (uint32_t i = 0; i < _max_requests; i++) {
	_active_requests++;
	get_route_list_next();
    }
}

// See RFC 1657 (BGP MIB) for full definitions of return values.

template <typename A>
void
PrintRoutes<A>::get_route_list_next_done(const XrlError& e,
					 const IPv4* peer_id,
					 const IPNet<A>* net,
					 const uint32_t *best_and_origin,
					 const vector<uint8_t>* aspath,
					 const A* nexthop,
					 const int32_t* med,
					 const int32_t* localpref,
					 const int32_t* atomic_agg,
					 const vector<uint8_t>* aggregator,
					 const int32_t* calc_localpref,
					 const vector<uint8_t>* attr_unknown,
					 const bool* valid,
					 const bool* /*unicast*/,
					 const bool* /*multicast*/)
{
    UNUSED(calc_localpref);
    UNUSED(attr_unknown);
    UNUSED(aspath);
    vector<uint8_t> community;

    if (e != XrlError::OKAY() || false == *valid) {
	_active_requests--;
	_done = true;
	return;
    }
    _count++;

    uint8_t best = (*best_and_origin)>>16;
    uint8_t origin = (*best_and_origin)&255;

    AsPath asp((const uint8_t*)(&((*aspath)[0])), aspath->size());
    print_route(*peer_id, *net, best, origin, asp, *nexthop, *med, *localpref, *atomic_agg, *aggregator, community);

    if (!_done)
    	get_route_list_next();
}

template <typename A>
void
PrintRoutes<A>::get_route_list_bulk_start_done(const XrlError& e,
					  const uint32_t* token)
{
    if (e != XrlError::OKAY()) {
	//fprintf(stderr, "Failed to get peer list start\n");
	if (_prev_no_bgp == false)
	    printf("\n\nBGP is not configured\n");
	_prev_no_bgp = true;
	_done = true;
	return;
    }
    _prev_no_bgp = false;

    switch(_verbose) {
    case SUMMARY:
    case NORMAL:
	printf("Status Codes: * valid route, > best route\n");
	printf("Origin Codes: i IGP, e EGP, ? incomplete\n\n");
	printf("   Prefix             Nexthop          BGP-ID    "
	       "       MED LP  AS-Path\n");
	printf("   ------             -------          ------    "
	       "       --- --- -------\n");
	break;
    case DETAIL:
	break;
    }

    _token = *token;
    for (uint32_t i = 0; i < MAX_BULK_REQUESTS; i++) {
	_active_requests++;
	switch (_mode) {
	case BULK_REQUEST:
	    get_route_list_bulk_next();
	    break;
	case FILE_REQUEST:
	    get_route_list_bulk_file_next();
	    break;
	case SINGLE_REQUEST:
	    printf("Internal error: expected bulk request mode - got single request.\n");
        };
    }
}

// See RFC 1657 (BGP MIB) for full definitions of return values.

class BufferReader
{
public:
    BufferReader(const vector<uint8_t>& data) : _f(data.begin()), _l(data.end()) {}

    int read(uint8_t *buf, size_t len) {
	size_t i = 0;
	while (_f != _l && i < len) {
	    buf[i++] = *(_f++);
        }
	return i;
    }
private:
    vector<uint8_t>::const_iterator _f;
    vector<uint8_t>::const_iterator _l;
};

template <class A>
void
PrintRoutes<A>::get_route_list_bulk_next_done(const XrlError& e,
					 const vector<uint8_t>* data,
					 const bool* has_more)
{
    RouteDumpRecord<A> dump_rec;
    uint8_t *var_data = new uint8_t[RT_REC_MAX_DATA];

    struct ResourceCleanup {
        ResourceCleanup (uint8_t *data, int *active_requests) :
		_data(data), _active_requests(active_requests) {}
	~ResourceCleanup () {
	    delete [] _data;
	    (*_active_requests)--;
	}
        uint8_t *_data;
	int *_active_requests;
    } cleanup(var_data, &_active_requests);

    if (e != XrlError::OKAY()) {
	printf("Error reading routes: %s\n", e.error_msg());
	_done = true;
	return;
    }

    memset((void *) &dump_rec, 0, sizeof(dump_rec));
    memset((void *) var_data, 0, RT_REC_MAX_DATA);
    BufferReader reader(*data);

    int rc = 0;
    while ((rc = reader.read((uint8_t *) &dump_rec, sizeof(dump_rec))) > 0) {
	IPv4 peer_id(dump_rec._peer_id);
	IPNet<A> net(A(dump_rec._net.masked_addr), dump_rec._net.prefix_len);
	A nexthop(dump_rec._nexthop);
	uint32_t vd_size = dump_rec._aspath_size + dump_rec._aggregator_size + dump_rec._community_size
		+ dump_rec._attr_unknown_size;
	if (vd_size > 0) {
	    rc = reader.read((uint8_t *) var_data, vd_size);
	    if (rc <= 0) {
		printf("Error reading routes: %s\n", strerror(errno));
		_done = true;
		return;
	    }
    	    AsPath asp((const uint8_t *) var_data, dump_rec._aspath_size);
            vector<uint8_t> agg((uint8_t *) (var_data + dump_rec._aspath_size), 
		    (uint8_t *) (var_data + dump_rec._aspath_size + dump_rec._aggregator_size));
            vector<uint8_t> community((uint8_t *) (var_data + dump_rec._aspath_size + dump_rec._aggregator_size), 
		    (uint8_t *) (var_data + dump_rec._aspath_size + dump_rec._aggregator_size + dump_rec._community_size));
            print_route(peer_id, net, dump_rec._best, dump_rec._origin, asp, nexthop,
                    dump_rec._med, dump_rec._localpref, dump_rec._atomic_agg, agg, community);
	}
	else {
	    AsPath asp;
	    vector<uint8_t> agg;
	    vector<uint8_t> community;
            print_route(peer_id, net, dump_rec._best, dump_rec._origin, asp, nexthop,
                    dump_rec._med, dump_rec._localpref, dump_rec._atomic_agg, agg, community);
	}

        memset((void *) &dump_rec, 0, sizeof(dump_rec));
        memset((void *) var_data, 0, RT_REC_MAX_DATA);
    }

    if (!_done && *has_more) {
    	get_route_list_bulk_next();
	_active_requests++;
    }
    else {
	_done = true;
    }
}

template <class A>
void
PrintRoutes<A>::get_route_list_bulk_file_next_done(const XrlError& e,
					 const string* filename,
					 const bool* has_more)
{
    RouteDumpRecord<A> dump_rec;
    uint8_t *var_data = new uint8_t[RT_REC_MAX_DATA];

    struct ResourceCleanup {
        ResourceCleanup (const string& filename, uint8_t *data, int *active_requests) :
		_filename(filename), _data(data), _active_requests(active_requests) {}
	~ResourceCleanup () {
	    delete [] _data;
	    unlink(_filename.c_str());
	    (*_active_requests)--;
	}
	string _filename;
        uint8_t *_data;
	int *_active_requests;
    } cleanup(*filename, var_data, &_active_requests);

    if (e != XrlError::OKAY()) {
	_done = true;
	return;
    }

    int fd = open(filename->c_str(), O_RDONLY | O_EXCL);
    if (fd < 0) {
	_done = true;
	return;
    }

    memset((void *) &dump_rec, 0, sizeof(dump_rec));
    memset((void *) var_data, 0, RT_REC_MAX_DATA);

    int rc = 0;
    while ((rc = read(fd, (void *) &dump_rec, sizeof(dump_rec))) > 0) {
	IPv4 peer_id(dump_rec._peer_id);
	IPNet<A> net(A(dump_rec._net.masked_addr), dump_rec._net.prefix_len);
	A nexthop(dump_rec._nexthop);
	uint32_t vd_size = dump_rec._aspath_size + dump_rec._aggregator_size + dump_rec._community_size
		+ dump_rec._attr_unknown_size;
	if (vd_size > 0) {
	    rc = read(fd, (void *) var_data, vd_size);
	    if (rc <= 0) {
		printf("Error reading routes: %s\n", strerror(errno));
		_done = true;
		close(fd);
		return;
	    }
    	    AsPath asp((const uint8_t *) var_data, dump_rec._aspath_size);
            vector<uint8_t> agg((uint8_t *) (var_data + dump_rec._aspath_size), 
		    (uint8_t *) (var_data + dump_rec._aspath_size + dump_rec._aggregator_size));
            vector<uint8_t> community((uint8_t *) (var_data + dump_rec._aspath_size + dump_rec._aggregator_size), 
		    (uint8_t *) (var_data + dump_rec._aspath_size + dump_rec._aggregator_size + dump_rec._community_size));
            print_route(peer_id, net, dump_rec._best, dump_rec._origin, asp, nexthop,
                    dump_rec._med, dump_rec._localpref, dump_rec._atomic_agg, agg, community);
	}
	else {
	    AsPath asp;
	    vector<uint8_t> agg;
	    vector<uint8_t> community;
            print_route(peer_id, net, dump_rec._best, dump_rec._origin, asp, nexthop,
                    dump_rec._med, dump_rec._localpref, dump_rec._atomic_agg, agg, community);
	}

        memset((void *) &dump_rec, 0, sizeof(dump_rec));
        memset((void *) var_data, 0, RT_REC_MAX_DATA);
    }

    close(fd);
    if (!_done && *has_more) {
    	get_route_list_bulk_file_next();
	_active_requests++;
    }
    else {
	_done = true;
    }
}

template <typename A>
void
PrintRoutes<A>::print_route(const IPv4& peer_id,
                                         const IPNet<A>& net,
                            		 uint8_t best,
                            		 uint8_t origin,
                            		 const AsPath asp,
                                         const A& nexthop,
                                         const int32_t& med,
                                         const int32_t& localpref,
                                         const int32_t& atomic_agg,
                            		 const vector<uint8_t>& aggregator,
                            		 const vector<uint8_t>& community)
{
    switch(_verbose) {
    case SUMMARY:
    case NORMAL:
	//XXX this should be used to indicate a route is valid
        printf("*");

        switch (best) {
        case 1:
            printf(" ");
            break;
        case 2:
            printf(">");
            break;
        default:
            printf("?");
        }

	char med_str[10], lp_str[10];
	sprintf(med_str, "%3d", med);
	sprintf(lp_str, "%3d", localpref);
        printf(" %-18s %-16s %-16s %-3s %-3s %s ", net.str().c_str(),
               nexthop.str().c_str(),
               peer_id.str().c_str(),
	       ((INVALID != med)? med_str: "   "), ((INVALID != localpref)? lp_str: "   "),
               asp.short_str().c_str());

        switch (origin) {
        case IGP:
            printf("i\n");
            break;
        case EGP:
            printf("e\n");
            break;
        case INCOMPLETE:
            printf("?\n");
            break;
        default:
            printf ("BAD ORIGIN\n");
            break;
        }
        break;
    case DETAIL:
        printf("%s\n", cstring(net));
        printf("\tFrom peer: %s\n", cstring(peer_id));
        printf("\tRoute: ");
        switch (best) {
        case 1:
            printf("Not Used\n");
            break;
        case 2:
            printf("Winner\n");
            break;
        default:
            printf("UNKNOWN\n");
        }

        printf("\tOrigin: ");
        switch (origin) {
        case IGP:
            printf("IGP\n");
            break;
        case EGP:
            printf("EGP\n");
            break;
        case INCOMPLETE:
            printf("INCOMPLETE\n");
            break;
        default:
            printf("BAD ORIGIN\n");
            break;
        }

        printf("\tAS Path: %s\n", asp.short_str().c_str());
        printf("\tNexthop: %s\n", cstring(nexthop));
        if (INVALID != med)
            printf("\tMultiple Exit Discriminator: %d\n", med);
        if (INVALID != localpref)
            printf("\tLocal Preference: %d\n", localpref);
        if (2 == atomic_agg)
            printf("\tAtomic Aggregate: Less Specific Route Selected\n");
#if	0
	printf("\tAtomic Aggregate: ");
	switch (atomic_agg) {
	case 1:
	    printf("Less Specific Route Not Selected\n");
	    break;
	case 2:
	    printf("Less Specific Route Selected\n");
	    break;
	default:
	    printf("UNKNOWN\n");
	    break;
	}
#endif
        if (!aggregator.empty()) {
            XLOG_ASSERT(6 == aggregator.size());
            A agg(&(aggregator[0]));
            AsNum asnum(&(aggregator[4]));

            printf("\tAggregator: %s %s\n", cstring(agg), cstring(asnum));
        }

        if (!community.empty() && community[0] > 0) {
            size_t size = 1 + community[0] * 4;
            XLOG_ASSERT(size == community.size());
 	    printf("\tCommunity: ");
            const uint8_t *p = &community[1];
            for (int i = 0; i < community[0]; i++, p += 4) {
		uint32_t val;
		memcpy(&val, p, 4);

		switch (val) {
		case NO_EXPORT:
		    printf("NO_EXPORT ");
		    break;
		case NO_ADVERTISE:
		    printf("NO_ADVERTISE ");
		    break;
		case NO_EXPORT_SUBCONFED:
		    printf("NO_EXPORT_SUBCONFED ");
		    break;
		default:
		    printf("%d:%d ", XORP_UINT_CAST(((val >> 16) & 0xffff)), XORP_UINT_CAST((val & 0xffff)));
		}
	    }
 	    printf("\n");
        }
        break;
    }

    _count++;
}

template <typename A>
void
PrintRoutes<A>::timer_expired()
{
    _done = true;
}

// ----------------------------------------------------------------------------
// Template Instantiations

template class PrintRoutes<IPv4>;
template class PrintRoutes<IPv6>;
