// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-

// Copyright (c) 2001-2007 International Computer Science Institute
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software")
// to deal in the Software without restriction, subject to the conditions
// listed in the XORP LICENSE file. These conditions include: you must
// preserve this copyright notice, and you cannot mention the copyright
// holders in advertising related to the Software without their permission.
// The Software is provided WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED. This
// notice is a summary of the XORP LICENSE file; the license in that file is
// legally binding.

// $XORP$

#ifndef __BGP_ROUTE_TABLE_RIBOUT_HH__
#define __BGP_ROUTE_TABLE_RIBOUT_HH__

#include <set>
#include <list>

#include "bgp_module.h"
#include "libxorp/xlog.h"
#include "libxorp/eventloop.hh"
#include "route_table_base.hh"
#include "route_queue.hh"
#include "parameter.hh"


#define QUEUE_FLAG_HASH_BITS    16
#define QUEUE_FLAG_HASH_MASK    0xffff
#define QUEUE_FLAG_HASH_SIZE    (1 << QUEUE_FLAG_HASH_BITS)

#define QUEUE_FLAG_HASH(x)      ((((unsigned int) (x) >> \
				    (32 - QUEUE_FLAG_HASH_BITS)) \
                                  ^ (unsigned int) (x)) & QUEUE_FLAG_HASH_MASK)

template<class A>
class RibOutTable : public BGPRouteTable<A>  {
public:
    RibOutTable(string tablename,
		Safi safi,
		BGPRouteTable<A> *parent,
		PeerHandler *peer);
    ~RibOutTable();
    void print_queue(const list<const RouteQueueEntry<A> *>& queue) const;
    int add_route(const InternalMessage<A> &rtmsg,
		  BGPRouteTable<A> *caller);
    int replace_route(const InternalMessage<A> &old_rtmsg,
		      const InternalMessage<A> &new_rtmsg,
		      BGPRouteTable<A> *caller);
    int delete_route(const InternalMessage<A> &rtmsg, 
		     BGPRouteTable<A> *caller);

    int push(BGPRouteTable<A> *caller);
    const SubnetRoute<A> *lookup_route(const IPNet<A> &net,
				       uint32_t& genid) const;

    RouteTableType type() const {return RIB_OUT_TABLE;}
    string str() const;
    int dump_entire_table() { abort(); return 0; }

    /* mechanisms to implement flow control in the output plumbing */
    void wakeup();
    bool pull_next_route();

    /* output_no_longer_busy is called asynchronously by the peer
       handler when the state of the output queue goes from busy to
       non-busy.  When the output queue has been busy we will have
       signalled back upstream to stop further routes arriving, so we
       need to go and retrieve the queued routes */
    void output_no_longer_busy();

    bool get_next_message(BGPRouteTable<A> */*next_table*/) {
	abort();
	return false;
    }

    void reschedule_self();

    void peering_went_down(const PeerHandler *peer, uint32_t genid,
			   BGPRouteTable<A> *caller);
    void peering_down_complete(const PeerHandler *peer, uint32_t genid,
			       BGPRouteTable<A> *caller);
    void peering_came_up(const PeerHandler *peer, uint32_t genid,
			 BGPRouteTable<A> *caller);
private:
    //the queue that builds, prior to receiving a push, so we can
    //send updates to our peers atomically
    list <const RouteQueueEntry<A> *> _queue;

    // Array of flags indicating that queue may hold a matching
    // route entry.  When route is added to queue, route is hashed to
    // an index into _queue_flag and corresponding flag is set.  When
    // new route arrives, it too is hashed into index into _queue_flag
    // and corresponding flag is checked.  If that flag is true, then
    // queue might hold matching route, so the queue must be searched
    // to try to find it.  If flag is not true, then
    // queue does not hold matching route, and the queue does not need
    // to be searched.  Since actual matches are rare, this approach
    // eliminates most queue searches.  We use a "char" as the array element,
    // which is effectively used as a boolean, to be as efficient as 
    // possible with space while avoiding bit manipulation.
    unsigned char _queue_flag[QUEUE_FLAG_HASH_SIZE];

    PeerHandler *_peer;
    bool _peer_busy;
    bool _peer_is_up;
    XorpTask _pull_routes_task;

    int _queue_flag_is_set(const IPNet<A>& net);
    void _set_queue_flag(const IPNet<A>& net);
};

#endif // __BGP_ROUTE_TABLE_RIBOUT_HH__
