// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-

// Copyright (c) 2001-2007 International Computer Science Institute
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software")
// to deal in the Software without restriction, subject to the conditions
// listed in the XORP LICENSE file. These conditions include: you must
// preserve this copyright notice, and you cannot mention the copyright
// holders in advertising related to the Software without their permission.
// The Software is provided WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED. This
// notice is a summary of the XORP LICENSE file; the license in that file is
// legally binding.

// $XORP$

#ifndef __BGP_ROUTE_TABLE_READER_HH__
#define __BGP_ROUTE_TABLE_READER_HH__

#include "libxorp/xorp.h"
#include "libxorp/ipnet.hh"
#include "bgp_trie.hh"

#include <map>

#if defined(HAVE_REGEX_H)
#include <regex.h>
#elif defined(HAVE_PCREPOSIX_H)
#include <pcre.h>
#include <pcreposix.h>
#else
#error "No header defining regex_t is present."
#endif


enum FieldCode {
    FIELD_ASPATH = 1,
    FIELD_COMMUNITY = 2
};


template <class A>
class ReaderIxTuple {
public:
    ReaderIxTuple(const IPv4& peer_id, bool one_shot);
    virtual ~ReaderIxTuple() {}
    const A& masked_addr() const {return _net.masked_addr();}
    uint32_t prefix_len() const {return _net.prefix_len();}
    IPNet<A> net() const {return _net;}
    bool is_one_shot() const { return _one_shot; }
    virtual bool is_consistent() const = 0;
    const IPv4& peer_id() const {return _peer_id;}
    bool operator<(const ReaderIxTuple& them) const;
protected:
    IPNet<A> _net;
    IPv4 _peer_id;
    bool _one_shot;
};

template <class A>
class RouteTableReader {
public:
    virtual ~RouteTableReader() {}
    virtual bool get_next(const SubnetRoute<A>*& route, IPv4& peer_id) = 0;
};

///////////////////////////////////////////////////////////////////

template <class A>
class RibInTable;

template <class A>
class RibInIxTuple : public ReaderIxTuple<A> {
public:
    typedef typename BgpTrie<A>::iterator trie_iterator;
    RibInIxTuple(const IPv4& peer_id,
		  trie_iterator route_iter, 
		  const RibInTable<A>* _ribin,
		  bool one_shot = false);
    virtual bool is_consistent() const;
    const RibInTable<A>* table() const {return _table;}
    trie_iterator& route_iterator()
    {
	return _route_iter;
    }
private:
    trie_iterator _route_iter;
    const RibInTable<A>* _table;
};

template <class A>
class RibInTableReader : public RouteTableReader<A> {
public:
    typedef typename BgpTrie<A>::iterator trie_iterator;
    RibInTableReader(const list<RibInTable<A>*>& ribins,
		     const IPNet<A>& prefix);
    virtual bool get_next(const SubnetRoute<A>*& route, IPv4& peer_id);
private:
    set <RibInIxTuple<A> > _peer_readers;
};

///////////////////////////////////////////////////////////////////

template <class A>
class DampingTable;

template <class A>
class DampRoute;

template <class A>
class DampingIxTuple : public ReaderIxTuple<A> {
public:
    typedef typename RefTrie<A, DampRoute<A> >::iterator trie_iterator;
    DampingIxTuple(const IPv4& peer_id,
		  trie_iterator route_iter, 
		  const DampingTable<A>* damping,
		  bool one_shot = false);
    virtual bool is_consistent() const;
    const DampingTable<A>* table() const {return _table;}
    trie_iterator& route_iterator()
    {
	return _route_iter;
    }
private:
    IPNet<A> _net;
    IPv4 _peer_id;
    trie_iterator _route_iter;
    const DampingTable<A>* _table;
};

template <class A>
class DampingTableReader : public RouteTableReader<A> {
public:
    typedef typename RefTrie<A, DampRoute<A> >::iterator trie_iterator;
    DampingTableReader(const list <DampingTable<A>*>& dampings,
		     const IPNet<A>& prefix);
    virtual bool get_next(const SubnetRoute<A>*& route, IPv4& peer_id);
private:
    set<DampingIxTuple<A> > _peer_readers;
};

///////////////////////////////////////////////////////////////////

template <class A>
class DecisionTable;

template <class A>
class DecisionData;

template <class A>
class DecisionTableReader : public RouteTableReader<A> {
public:
    typedef typename RefTrie<A, DecisionData<A> >::iterator trie_iterator;
    typedef typename DecisionData<A>::iterator route_iterator;
    DecisionTableReader(const DecisionTable<A>* table, const IPv4& peer_id,
		     const IPNet<A>& prefix, const string& regex, FieldCode field);
    virtual ~DecisionTableReader();
    virtual bool get_next(const SubnetRoute<A>*& route, IPv4& peer_id);
private:
    trie_iterator _trie_iter;
    route_iterator _route_iter;
    const DecisionTable<A>* _table;
    IPv4 _peer_id;
    string _regex;
    FieldCode _field;
    bool _one_shot;
    regex_t* _re;
};

#endif // __BGP_ROUTE_TABLE_READER_HH__
