// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-

// Copyright (c) 2001-2007 International Computer Science Institute
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software")
// to deal in the Software without restriction, subject to the conditions
// listed in the XORP LICENSE file. These conditions include: you must
// preserve this copyright notice, and you cannot mention the copyright
// holders in advertising related to the Software without their permission.
// The Software is provided WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED. This
// notice is a summary of the XORP LICENSE file; the license in that file is
// legally binding.

#ident "$XORP$"

#include "bgp_module.h"
#include "libxorp/xlog.h"
#include "route_table_reader.hh"
#include "route_table_ribin.hh"
#include "route_table_damping.hh"
#include "route_table_decision.hh"
#include "peer_handler.hh"
#include "bgp.hh"

template <class A>
ReaderIxTuple<A>::ReaderIxTuple(const IPv4& peer_id, bool one_shot) : _peer_id(peer_id), _one_shot(one_shot)
{
}

template <class A>
bool
ReaderIxTuple<A>::operator<(const ReaderIxTuple& them) const 
{
    if (masked_addr() < them.masked_addr()) {
	return true;
    }
    if (them.masked_addr() < masked_addr()) {
	return false;
    }
#if 0
    //The MIB requires it to be least-specific first
    if (prefix_len() < them.prefix_len()) {
	return true;
    }
    if (them.prefix_len() < _prefix_len()) {
	return false;
    }
#else
    //XXX But we don't have a Trie iterator for that, so we do most
    //specific first for now.
    if (prefix_len() < them.prefix_len()) {
	return false;
    }
    if (them.prefix_len() < prefix_len()) {
	return true;
    }
#endif
    if (_peer_id < them.peer_id()) {
	return true;
    }
    if (them.peer_id() < _peer_id) {
	return false;
    }
    return false;
}

template <class A>
RibInIxTuple<A>::RibInIxTuple(const IPv4& peer_id,
				trie_iterator route_iter, 
				const RibInTable<A>* ribin,
				bool one_shot)
    : ReaderIxTuple<A>(peer_id, one_shot), _route_iter(route_iter), _table(ribin) 
{
    ReaderIxTuple<A>::_net = _route_iter.key();
}

template <class A>
bool
RibInIxTuple<A>::is_consistent() const {
    if (_route_iter == _table->trie().end()) {
	// the route we were pointing at has been deleted, and there
	// are no routes after this in the trie.
	return false;
    }
    if (_route_iter.key() != ReaderIxTuple<A>::_net) {
	// the route we were pointing at has been deleted, and the
	// iterator now points to a later subnet in the trie.
	return false;
    }
    return true;
}

template <class A>
RibInTableReader<A>::RibInTableReader(const list <RibInTable<A>*>& ribins,
				      const IPNet<A>& prefix)
{
    typename list <RibInTable<A>*>::const_iterator i;
    if (0 == prefix.prefix_len()) {
	for(i = ribins.begin(); i != ribins.end(); i++) {
	    trie_iterator ti = (*i)->trie().begin();
	    if (ti != (*i)->trie().end()) {
		IPv4 peer_id = (*i)->peer_handler()->id();
		_peer_readers.insert(RibInIxTuple<A>(peer_id, ti, (*i)));
	    }
	}
    }
    else {
	for(i = ribins.begin(); i != ribins.end(); i++) {
	    trie_iterator ti = (*i)->trie().find(prefix);
	    if (ti != (*i)->trie().end()) {
		IPv4 peer_id = (*i)->peer_handler()->id();
		_peer_readers.insert(RibInIxTuple<A>(peer_id, ti, (*i), true /* one shot */));
	    }
	}
    }
}

template <class A>
bool
RibInTableReader<A>::get_next(const SubnetRoute<A>*& route, IPv4& peer_id) 
{
    typename set <RibInIxTuple<A> >::iterator i;
    while (1) {
	i = _peer_readers.begin();
	if (i == _peer_readers.end()) {
	    return false;
	}
	RibInIxTuple<A> reader = *i;
	
	if (i->is_consistent()) {
	    //return the route and the peer_id from the reader
	    route = &(reader.route_iterator().payload());
	    peer_id = reader.peer_id();

	    //if necessary, prepare this peer's reader for next time
	    if (!reader.is_one_shot()) {
		reader.route_iterator()++;
		if (reader.route_iterator() != reader.table()->trie().end()) {
			_peer_readers.insert(RibInIxTuple<A>(reader.peer_id(),
							  reader.route_iterator(),
							  reader.table()));
		}
	    }

	    //remove the obsolete reader
	    _peer_readers.erase(i);
	    //delete reader;
	    if (peer_id == IPv4::ZERO())
		continue;
	    return true;
	}

	//the iterator was inconsistent, so we need to re-insert a
	//consistent version into the set and try again.
	if (reader.route_iterator() != reader.table()->trie().end()) {
	    _peer_readers.insert(RibInIxTuple<A>(reader.peer_id(),
						      reader.route_iterator(),
						      reader.table()));
	}
	//remove the obsolete reader
	_peer_readers.erase(i);
	//delete reader;
    }
}

template <class A>
DampingIxTuple<A>::DampingIxTuple(const IPv4& peer_id,
				trie_iterator route_iter,
				const DampingTable<A>* damping,
				bool one_shot)
    : ReaderIxTuple<A>(peer_id, one_shot), _route_iter(route_iter), _table(damping) 
{
    ReaderIxTuple<A>::_net = _route_iter.key();
}

template <class A>
bool
DampingIxTuple<A>::is_consistent() const {
    if (_route_iter == _table->trie().end()) {
	// the route we were pointing at has been deleted, and there
	// are no routes after this in the trie.
	return false;
    }
    if (_route_iter.key() != _net) {
	// the route we were pointing at has been deleted, and the
	// iterator now points to a later subnet in the trie.
	return false;
    }
    return true;
}

template <class A>
DampingTableReader<A>::DampingTableReader(const list <DampingTable<A>*>& dampings,
				      const IPNet<A>& prefix)
{
    typename list <DampingTable<A>*>::const_iterator i;
    if (0 == prefix.prefix_len()) {
	for(i = dampings.begin(); i != dampings.end(); i++) {
	    trie_iterator ti = (*i)->trie().begin();
	    if (ti != (*i)->trie().end()) {
		IPv4 peer_id = (*i)->peer_handler()->id();
		_peer_readers.insert(DampingIxTuple<A>(peer_id, ti, (*i)));
	    }
	}
    }
    else {
	for(i = dampings.begin(); i != dampings.end(); i++) {
	    trie_iterator ti = (*i)->trie().find(prefix);
	    if (ti != (*i)->trie().end()) {
		IPv4 peer_id = (*i)->peer_handler()->id();
		_peer_readers.insert(DampingIxTuple<A>(peer_id, ti, (*i), true));
	    }
	}
    }
}

template <class A>
bool
DampingTableReader<A>::get_next(const SubnetRoute<A>*& route, IPv4& peer_id) 
{
    typename set <DampingIxTuple<A> >::iterator i;
    while (1) {
	i = _peer_readers.begin();
	if (i == _peer_readers.end()) {
	    return false;
	}
	DampingIxTuple<A> reader = *i;
	
	if (reader.is_consistent()) {
	    //return the route and the peer_id from the reader
	    route = reader.route_iterator().payload().route();
	    peer_id = reader.peer_id();

	    //if necessary, prepare this peer's reader for next time
	    if (!reader.is_one_shot()) {
		reader.route_iterator()++;
		if (reader.route_iterator() != reader.table()->trie().end()) {
			_peer_readers.insert(DampingIxTuple<A>(reader.peer_id(),
							  reader.route_iterator(),
							  reader.table()));
		}
	    }

	    //remove the obsolete reader
	    _peer_readers.erase(i);
	    //delete reader;
	    return true;
	}

	//the iterator was inconsistent, so we need to re-insert a
	//consistent version into the set and try again.
	if (reader.route_iterator() != reader.table()->trie().end()) {
	    _peer_readers.insert(DampingIxTuple<A>(reader.peer_id(),
						      reader.route_iterator(),
						      reader.table()));
	}
	//remove the obsolete reader
	_peer_readers.erase(i);
	//delete reader;
    }
}

////////////////////////////////////////////////////////////////////////

regex_t*
compile_regex(const string& regex)
{
    regex_t *re = new regex_t;
    int res = regcomp(re, regex.c_str(), REG_EXTENDED);
    if (res) {
	char tmp[128];
	string err;

	regerror(res, re, tmp, sizeof(tmp));
	regfree(re);

	err = "Unable to compile regex (" + regex;
	err += "): ";
	err += tmp;

	xorp_throw(BGPException, err);
    }
    return re;
}

template <class A>
DecisionTableReader<A>::DecisionTableReader(const DecisionTable<A>* table, const IPv4& peer_id,
	const IPNet<A>& prefix, const string& regex, FieldCode field) :
	_table(table), _peer_id(peer_id), _regex(regex), _field(field)
{
    if (0 == prefix.prefix_len()) {
	_trie_iter = table->trie().begin();
	_one_shot = false;
    }
    else {
	_trie_iter = table->trie().find(prefix);
	_one_shot = true;
    }

    if (_trie_iter != _table->trie().end())
	_route_iter = _trie_iter.payload().begin();

    // compile the regex
    if (_field != 0 && !_regex.empty())
	_re = compile_regex(_regex);
}

template <class A>
DecisionTableReader<A>::~DecisionTableReader()
{
    if (_field != 0 && _re) {
	regfree(_re);
	delete _re;
    }
}

template <class A>
bool
DecisionTableReader<A>::get_next(const SubnetRoute<A>*& route, IPv4& peer_id)
{
    while (_trie_iter != _table->trie().end()) {

	while (_route_iter != _trie_iter.payload().end()) {
	    peer_id = _route_iter->first;
	    route = _route_iter->second.route();
	    _route_iter++;

	    if (_field != 0 && _re) {
		if (_field == FIELD_ASPATH) {
		    string as_str = route->attributes()->aspath().short_str();
		    bool result = !regexec(_re, as_str.c_str(), 0, 0, 0);
		    if (!result) continue;
		}
		else if (_field == FIELD_COMMUNITY) {
		    string com_str = route->attributes()->community_att()->str();
		    bool result = !regexec(_re, com_str.c_str(), 0, 0, 0);
		    if (!result) continue;
		}
	    }

	    if (_peer_id == IPv4::ZERO() || peer_id == _peer_id)
		return true;
        }

	if (_one_shot)
	    _trie_iter = _table->trie().end();
	else
	    _trie_iter++;

	if (_trie_iter != _table->trie().end())
	    _route_iter = _trie_iter.payload().begin();
    }

    return false;
}

template class RibInTableReader<IPv4>;
template class RibInTableReader<IPv6>;
template class DampingTableReader<IPv4>;
template class DampingTableReader<IPv6>;
template class DecisionTableReader<IPv4>;
template class DecisionTableReader<IPv6>;
