// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
// vim:set sts=4 ts=8:


#ident "$XORP$"

//#define DEBUG_LOGGING
//#define DEBUG_PRINT_FUNCTION_NAME

// #define DEBUG_QUEUE

#include "bgp_module.h"
#include "libxorp/xlog.h"
#include "route_table_dupfilt.hh"

template<class A>
DupfiltTable<A>::DupfiltTable(string table_name,
			      Safi safi,
			      BGPRouteTable<A> *init_parent)
    : BGPRouteTable<A>("DupfiltTable-" + table_name, safi)
{
    this->_parent = init_parent;

    // Initialize stats.
    _stats_print_rate = 0;
    _queue_add_count = 0;
    _queue_del_count = 0;
    _queue_count = 0;
    _add_route_count = 0;
    _replace_route_count = 0;
    _delete_route_count = 0;
    _route_dump_count = 0;
    _push_count = 0;
    _queue_empty_count = 0;
    _hash_del_count = 0;

    for (int i = 0; i < DUPFILT_HASH_SIZE; i++)
	_hash[i] = NULL;
    _output_queue_head = NULL;
    _output_queue_tail = NULL;
}

template<class A>
DupfiltTable<A>::~DupfiltTable()
{
}

template<class A>
int
DupfiltTable<A>::add_route(const InternalMessage<A> &rtmsg,
			   BGPRouteTable<A> *caller) 
{
    debug_msg("\n         %s\n caller: %s\n rtmsg: %p route: %p\n%s\n",
	      this->tablename().c_str(),
	      caller ? caller->tablename().c_str() : "NULL",
	      &rtmsg,
	      rtmsg.route(),
	      rtmsg.str().c_str());

    XLOG_ASSERT(caller == this->_parent);
    XLOG_ASSERT(rtmsg.route()->nexthop_resolved());

    _add_route_count++;
    debug_call_counts();

    DupfiltRouteQueueEntry<A> *match_entry;
    match_entry = hash_find_match(rtmsg);
    if (match_entry != NULL) {
	switch(match_entry->op()) {
	case RTQUEUE_OP_ADD:
	    // Should never see an Add followed by a second Add of the
	    // same route.
	    XLOG_FATAL("Duplicate add of same route");
	    break;

	case RTQUEUE_OP_DELETE:
	    // Found a previous delete of the route being put back with this
	    // Add.  We can just remove both commands from the pipeline.
	    hash_list_del(match_entry);
	    _hash_del_count++;
	    output_queue_del(match_entry);
	    delete (match_entry);
	    debug_queue_counts();
	    break;

	case RTQUEUE_OP_REPLACE_OLD:
	    // We should not see a new Add command for a route that was
	    // previously replaced without the replacement route first
	    // being deleted.  That would turn the "Replace Old" command
	    // into a "Delete".  Hence this sequence should never happen.
	    XLOG_FATAL("Add after Replace_Old of same route");
	    break;

	case RTQUEUE_OP_REPLACE_NEW:
	    // We should never see a route added via "Replace New" added
	    // again without an intervening "Replace" or "Delete"
	    XLOG_FATAL("Add after Replace_New of same route");
	    break;

	default:
	    // Invalid opcode found in hash table 
	    XLOG_FATAL("Invalid opcode %d found in hash table", 
		       match_entry->op());
	    break;
	}
    } else {
	// No match in hash table, so this message goes onto the queue
	add_to_queue(RTQUEUE_OP_ADD, rtmsg);
	do_wakeup();
    }

    return ADD_USED;
}

template<class A>
int
DupfiltTable<A>::delete_route(const InternalMessage<A> &rtmsg,
			     BGPRouteTable<A> *caller) 
{
    debug_msg("\n         %s\n caller: %s\n rtmsg: %p route: %p\n%s\n",
	      this->tablename().c_str(),
	      caller ? caller->tablename().c_str() : "NULL",
	      &rtmsg,
	      rtmsg.route(),
	      rtmsg.str().c_str());
    
    XLOG_ASSERT(caller == this->_parent);
    XLOG_ASSERT(rtmsg.route()->nexthop_resolved());

    _delete_route_count++;
    debug_call_counts();

    DupfiltRouteQueueEntry<A> *match_entry;
    match_entry = hash_find_match(rtmsg);
    if (match_entry != NULL) {
	switch (match_entry->op()) {
	case RTQUEUE_OP_ADD:
	    // Found the "Add" of the route that is now being deleted.
	    // That "add" can be simply discarded, and this "Delete" message
	    // can be ignored.
	    hash_list_del(match_entry);
	    _hash_del_count++;
	    output_queue_del(match_entry);
	    delete (match_entry);
	    debug_queue_counts();
	    break;

	case RTQUEUE_OP_DELETE:
	    // Should never see consecutive deletes of the same route
	    XLOG_FATAL("Duplicate delete");
	    break;
	    
	case RTQUEUE_OP_REPLACE_OLD:
	    // Should never see a delete after a replace_old either
	    XLOG_FATAL("Delete after Replace_Old of same route");
	    break;
	    
	case RTQUEUE_OP_REPLACE_NEW:
	    // The "new" route of the old/new pair has now been deleted.
	    // We can translate that entry into a delete of the "old" route.
	    //
	    // First find the "old" route.  It had better be the
	    // DupfiltRouteQueueEntry immediately preceeding the "new" entry.
	    DupfiltRouteQueueEntry<A> *old_entry;
	    old_entry = match_entry->queue_next();
	    XLOG_ASSERT(old_entry != NULL);

	    // Next, pluck them both off their hash and fifo queues,
	    // and delete the "new" entry that we matched because it is
	    // being eliminated from the pipeline.
	    hash_list_del(match_entry);
	    _hash_del_count++;
	    output_queue_del(match_entry);
	    hash_list_del(old_entry);
	    output_queue_del(old_entry);
	    delete(match_entry);

	    // Now we transform the "old" route into a delete by just
	    // changing its opcode, then put it back into the queues.
	    old_entry->set_op(RTQUEUE_OP_DELETE);
	    output_queue_push(old_entry);
	    hash_list_insert(old_entry);

	    // And we don't use the delete rtmsg that came in with this call.
	    break;

	default:
	    // Invalid opcode in DupfiltRouteQueueEntry found in hash table
	    XLOG_FATAL("Invalid opcode %d found in hash table",
		       match_entry->op());
	    break;
	}
    } else {
	// No matching route found in hash table.  Go ahead and
	// add this one to the output queue and hash table.
	add_to_queue(RTQUEUE_OP_DELETE, rtmsg);
	do_wakeup();
    }
    
    return 0;
}

template<class A>
int
DupfiltTable<A>::replace_route(const InternalMessage<A> &old_rtmsg,
			      const InternalMessage<A> &new_rtmsg,
			      BGPRouteTable<A> *caller) 
{
    debug_msg("\n         %s\n"
	      "caller: %s\n"
	      "old rtmsg: %p new rtmsg: %p "
	      "old route: %p"
	      "new route: %p"
	      "old: %s\n new: %s\n",
	      this->tablename().c_str(),
	      caller->tablename().c_str(),
	      &old_rtmsg,
	      &new_rtmsg,
	      old_rtmsg.route(),
	      new_rtmsg.route(),
	      old_rtmsg.str().c_str(),
	      new_rtmsg.str().c_str());

    XLOG_ASSERT(caller == this->_parent);
    XLOG_ASSERT(old_rtmsg.route()->nexthop_resolved());
    XLOG_ASSERT(new_rtmsg.route()->nexthop_resolved());

    _replace_route_count++;
    debug_call_counts();

    // We look only for the "old" route in the hash table.
    DupfiltRouteQueueEntry<A> *match_entry;
    match_entry = hash_find_match(old_rtmsg);
    if (match_entry != NULL) {
	switch (match_entry->op()) {
	case RTQUEUE_OP_ADD:
	    // Previously added route is now being removed via replace old/new.
	    // We delete the old "Add", then transform the "replace old/new"
	    // into a simple "Add" of the new route.  
	    // Delete the matched "Add":
	    hash_list_del(match_entry);
	    _hash_del_count++;
	    output_queue_del(match_entry);
	    delete(match_entry);
	    debug_queue_counts();

	    // Then add the "new" message to the queue, but as an "Add":
	    add_to_queue(RTQUEUE_OP_ADD, new_rtmsg);	    
	    do_wakeup();
	    break;

	case RTQUEUE_OP_DELETE:
	    // Should never occur.
	    XLOG_FATAL("Delete after Replace_Old of same route");
	    break;
	    
	case RTQUEUE_OP_REPLACE_OLD:
	    // Should never happen.
	    XLOG_FATAL("Duplicate Replace_Old");
	    break;
	    
	case RTQUEUE_OP_REPLACE_NEW:
	    // A route previously added via "replace old/new" is now
	    // being replaced.  So, the sequence of events that have
	    // occurred are:  1) Replace Old X, New Y; 2) Replace Old Y, 
	    // New Z.  This can be transformed into:  Replace Old X, new Z,
	    // unless X happens to be the same route as Z.  In that
	    // case, both pairs can be eliminated from the pipeline, and the
	    // original route X remains in place.
	    //
	    // First we have to find the "Old" route that goes with
	    // the matched "New" route:
	    DupfiltRouteQueueEntry<A> *old_entry;
	    old_entry = match_entry->queue_next();
	    XLOG_ASSERT(old_entry != NULL);

	    if (old_entry->route() != new_rtmsg.route()) {
		// Delete "New" entry from the queue and hash:
		hash_list_del(match_entry);
		_hash_del_count++;
		output_queue_del(match_entry);
		delete(match_entry);
		debug_queue_counts();

		// Move "Old" entry to the tail of the queue.  Since the
		// route it points to does not change, its hash table entry
		// does not change:
		output_queue_del(old_entry);
		output_queue_push(old_entry);

		// Now push the new "New" entry for this "replace old/new"
		// pair:
		add_to_queue(RTQUEUE_OP_REPLACE_NEW, new_rtmsg);
		do_wakeup();
	    } else {
		// Delete both the "Old" and "New" matched entries:
		hash_list_del(match_entry);
		_hash_del_count++;
		output_queue_del(match_entry);
		delete(match_entry);
		debug_queue_counts();

		hash_list_del(old_entry);
		_hash_del_count++;
		output_queue_del(old_entry);
		delete(old_entry);
		debug_queue_counts();

		// And the Old/New pair passed in are not used
	    }
	    break;

	default:
	    // Invalid opcode in DupfiltRouteQueueEntry found in hash table
	    XLOG_FATAL("Invalid opcode %d found in hash table",
		       match_entry->op());
	    break;
	}
    } else {
	add_to_queue(RTQUEUE_OP_REPLACE_OLD, old_rtmsg);
	add_to_queue(RTQUEUE_OP_REPLACE_NEW, new_rtmsg);
	do_wakeup();
    }

    return ADD_USED;
}

template<class A>
int
DupfiltTable<A>::route_dump(const InternalMessage<A> &rtmsg,
			   BGPRouteTable<A> *caller,
			   const PeerHandler *dump_peer) 
{
    XLOG_ASSERT(caller == this->_parent);
    XLOG_ASSERT(rtmsg.route()->nexthop_resolved());

    _route_dump_count++;
    debug_call_counts();

    int ret = this->_next_table->route_dump(rtmsg, (BGPRouteTable<A>*)this,
					    dump_peer);
    return ret;
}

template<class A>
int
DupfiltTable<A>::push(BGPRouteTable<A> *caller) 
{
    debug_msg("Push\n");
    XLOG_ASSERT(caller == this->_parent);

    _push_count++;
    debug_call_counts();
    add_push_to_queue();
    debug_queue_counts();
    do_wakeup();

    //int ret = this->_next_table->push((BGPRouteTable<A> *) this);
    return 0;
}

template<class A>
const SubnetRoute<A>*
DupfiltTable<A>::lookup_route(const IPNet<A> &net, uint32_t& genid) const 
{
    return this->_parent->lookup_route(net, genid);
}

template<class A>
string
DupfiltTable<A>::str() const 
{
    string s = "DupfiltTable<A>" + this->tablename();
    return s;
}


#define MSG_BATCH_SIZE 1

template<class A>
bool
DupfiltTable<A>::get_next_message(BGPRouteTable<A> *next_table) 
{
    debug_msg("next table: %s\n", next_table->tablename().c_str());

    DupfiltRouteQueueEntry<A> *entry;
    entry = output_queue_pop();
    if (entry == NULL) {
	_queue_empty_count++;
	return false;
    }

    RouteQueueOp operation = entry->op();
    if (operation == RTQUEUE_OP_ADD) {
	InternalMessage<A> rtmsg(entry->route(),
				 entry->origin_peer(),
				 entry->genid());
	if (entry->push())
	    rtmsg.set_push();
	this->_next_table->add_route(rtmsg, (BGPRouteTable<A> *) this);
    } else if (operation == RTQUEUE_OP_DELETE) {
	InternalMessage<A> rtmsg(entry->route(),
				 entry->origin_peer(),
				 entry->genid());
	if (entry->push())
	    rtmsg.set_push();
	this->_next_table->delete_route(rtmsg, (BGPRouteTable<A> *) this);
    } else if (operation == RTQUEUE_OP_REPLACE_OLD) {
	InternalMessage<A> old_rtmsg(entry->route(),
				     entry->origin_peer(),
				     entry->genid());
	if (entry->push())
	    old_rtmsg.set_push();
	const DupfiltRouteQueueEntry<A> *old_queue_entry = entry;

	entry = output_queue_pop();
	XLOG_ASSERT(entry != NULL);
	InternalMessage<A> new_rtmsg(entry->route(),
				     entry->origin_peer(),
				     entry->genid());
	if (entry->push())
	    new_rtmsg.set_push();

	this->_next_table->replace_route(old_rtmsg, new_rtmsg,
					 (BGPRouteTable<A> *) this);
	delete (old_queue_entry);
	debug_queue_counts();
    } else if (operation == RTQUEUE_OP_PUSH) {
	this->_next_table->push((BGPRouteTable<A> *) this);
    } else {
	XLOG_ERROR("Got op code %d\n", (int) operation);
	XLOG_ERROR("dupfilt queue: add %d, del %d, count %d\n",
		   _queue_add_count, _queue_del_count, _queue_count);
	XLOG_UNREACHABLE();
    }

    delete (entry);
    debug_queue_counts();

    return true;
}

template<class A>
void
DupfiltTable<A>::debug_call_counts()
{
    if ((_stats_print_rate > 0) &&
	(((_add_route_count + _replace_route_count + _delete_route_count +
	   _route_dump_count + _push_count) % _stats_print_rate) == 0))
	XLOG_ERROR("Dupfilt calls: add = %d, replace = %d, delete = %d, dump = %d, push = %d, wakeups = %d\n", 
		 _add_route_count, _replace_route_count, _delete_route_count,
		 _route_dump_count, _push_count, _wakeup_calls);
}

template<class A>
void
DupfiltTable<A>::debug_queue_counts()
{
    if ((_stats_print_rate > 0) &&
	(((_queue_add_count + _queue_del_count) % _stats_print_rate) == 0))
	XLOG_ERROR("dupfilt queue: add %d, del %d, count %d, empty count %d, hash_del_count %d, pull_del_count %d\n",
		   _queue_add_count, _queue_del_count, _queue_count,
		   _queue_empty_count, _hash_del_count, _pull_del_count);
}


// Generate a new DupfiltRouteQueueEntry for this rtmsg and add it to both
// the output queue and the hash list.
template<class A>
void
DupfiltTable<A>::add_to_queue(RouteQueueOp operation, 
			      const InternalMessage<A> &rtmsg)
{
    // Generate a new DupfiltRouteQueueEntry
    DupfiltRouteQueueEntry<A>* entry;
    entry = new DupfiltRouteQueueEntry<A>(rtmsg.route(), operation);
    entry->set_origin_peer(rtmsg.origin_peer());
    entry->set_genid(rtmsg.genid());
    if (rtmsg.push())
	entry->set_push(true);

    // Add it to the tail of the output queue
    output_queue_push(entry);
    if (rtmsg.changed())
	rtmsg.inactivate();

    // And insert at head hash chain of a hash list
    hash_list_insert(entry);
}


template<class A>
void
DupfiltTable<A>::add_push_to_queue()
{
    // Generate a new DupfiltRouteQueueEntry
    DupfiltRouteQueueEntry<A>* entry;
    entry = new DupfiltRouteQueueEntry<A>(RTQUEUE_OP_PUSH, NULL);

    // Add it to the tail of the output queue
    output_queue_push(entry);

    // Push messages don't go into hash lists
}

// Insert a DupfiltRouteQueueEntry onto a hash list
template<class A>
void
DupfiltTable<A>::hash_list_insert(DupfiltRouteQueueEntry<A> *entry)
{
    int index = DUPFILT_HASH(entry->route());
    DupfiltRouteQueueEntry<A> *next_entry;
    next_entry = _hash[index];
    entry->set_hash_next(next_entry);
    entry->set_hash_prev(NULL);
    if (next_entry != NULL)
	next_entry->set_hash_prev(entry);
    _hash[index] = entry;
}


// Delete a DupfiltRouteQueueEntry from a hash list
template<class A>
void
DupfiltTable<A>::hash_list_del(DupfiltRouteQueueEntry<A> *entry)
{
    DupfiltRouteQueueEntry<A> *prev = entry->hash_prev();
    DupfiltRouteQueueEntry<A> *next = entry->hash_next();

    if (next != NULL)
	next->set_hash_prev(prev);
    if (prev != NULL)
	prev->set_hash_next(next);
    else {
	// This is the first entry on the hash list.
	// Point the hash list head to whatever this was pointing to.
	// If this was the ONLY entry on the list, then hash list head
	// will be set to point to NULL.
	int index;
	index = DUPFILT_HASH(entry->route());
	_hash[index] = next;
    }
}

// Try to find a DupfiltRouteQueueEntry on a hash chain matching the route
// pointed to by the rtmsg passed in.  Return NULL if none found.
template<class A>
DupfiltRouteQueueEntry<A> *DupfiltTable<A>::hash_find_match(const InternalMessage<A> &rtmsg)
{
    int index = DUPFILT_HASH(rtmsg.route());
    DupfiltRouteQueueEntry<A> *entry = _hash[index];
    while (entry != NULL) {
	if (entry->route() == rtmsg.route())
	    return (entry);
	else
	    entry = entry->hash_next();
    }
    // no match
    return (NULL);
}


// Push a DupfiltRouteQueueEntry onto tail of output queue.
template<class A>
void
DupfiltTable<A>::output_queue_push(DupfiltRouteQueueEntry<A> *entry)
{
    entry->set_queue_next(_output_queue_tail);
    entry->set_queue_prev(NULL);
    if (_output_queue_tail != NULL)
	_output_queue_tail->set_queue_prev(entry);
    _output_queue_tail = entry;
    if (_output_queue_head == NULL)
	_output_queue_head = entry;
    _queue_add_count++;
    _queue_count++;
    debug_queue_counts();
}

// Pop a DupfiltRouteQueueEntry off of tail of output queue, and return
// that entry.
// Make sure we take it off its hash list too.
template<class A>
DupfiltRouteQueueEntry<A> *
DupfiltTable<A>::output_queue_pop()
{
    if (_output_queue_head == NULL)
	return NULL;
    DupfiltRouteQueueEntry<A> *entry;
    entry = _output_queue_head;

    // Delete from hash list as well
    if (entry->op() != RTQUEUE_OP_PUSH) {
	hash_list_del(entry);
    }

    // Delete from output queue
    output_queue_del(entry);

    _pull_del_count++;
    debug_queue_counts();
    return (entry);
}


// Delete a DupfiltRouteQueueEntry out of the middle of the output queue
template<class A>
void
DupfiltTable<A>::output_queue_del(DupfiltRouteQueueEntry<A> *entry)
{
    DupfiltRouteQueueEntry<A> *prev = entry->queue_prev();
    DupfiltRouteQueueEntry<A> *next = entry->queue_next();

    if (next != NULL)
	next->set_queue_prev(prev);
    else {
	// This was the entry at the head of the output queue
	_output_queue_head = prev;
    }

    if (prev != NULL)
	prev->set_queue_next(next);
    else {
	// This was the entry at the tail of the output queue
	_output_queue_tail = next;
    }
    _queue_del_count++;
    _queue_count--;
}

template<class A>
void
DupfiltTable<A>::do_wakeup()
{
    this->_next_table->wakeup();
    _wakeup_calls++;
}

template class DupfiltTable<IPv4>;
template class DupfiltTable<IPv6>;
