/* -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*- */
/*
 * Module: linux_tcpmd5.c
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) "YEAR" Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Tom Grennan
 * Date: 2007
 * Description: linux specific setsockopt() for TCP_MD5SIG
 *
 * **** End License ****
 *
 */
#include <errno.h>
#include <string.h>
#include <sys/socket.h>
#include <linux/in.h>
#include <linux/tcp.h>
#include "tcpmd5.h"

/*
 * If TCP_MD5SIG or TCP_MD5SIG_MAXKEYLEN are undefined, our compile
 * system headers may just be out of date with the target kernel. We
 * recreate the 2.6.20 definitions as the worst that may happen is an
 * ENOPROTOOPT.
 */
#if !defined(TCP_MD5SIG)
#  define TCP_MD5SIG 14
#endif

#if !defined(TCP_MD5SIG_MAXKEYLEN)
#  define TCP_MD5SIG_MAXKEYLEN 80
struct tcp_md5sig {
    struct __kernel_sockaddr_storage tcpm_addr;	/* address associated */
    unsigned short	__tcpm_pad1;			/* zero */
    unsigned short	tcpm_keylen;			/* key length */
    unsigned long	__tcpm_pad2;			/* zero */
    char		tcpm_key[TCP_MD5SIG_MAXKEYLEN];	/* key (binary) */
};
#endif

#if !defined(SOL_TCP)
#  define SOL_TCP IPPROTO_TCP
#endif

int
setsockopt_tcpmd5(int sock,
		  const struct sockaddr *saddr,
		  size_t slen,
		  const char *key)
{
    struct tcp_md5sig sig;
    int rval;

    memset(&sig, 0, sizeof sig);
    memcpy(&sig.tcpm_addr, saddr, slen);
    /*
     * a null or zero length key will disable the md5 signature
     */
    if (key && *key) {
	strncpy((char*)sig.tcpm_key, key, TCP_MD5SIG_MAXKEYLEN - 1);
	sig.tcpm_keylen = strlen((const char*)sig.tcpm_key);
    }
    rval = setsockopt(sock, SOL_TCP, TCP_MD5SIG,
		      (const void *)&sig, sizeof sig);
    /*
     * ignore ENOENT as just indicates that we cleared the md5 key
     * on a socket that hadnot had it set.
     */
    if (rval < 0 && errno == ENOENT && !(key && *key))
	rval = errno = 0;
    return rval;
}
